<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use JsonException;
use Symfony\Component\Filesystem\Filesystem;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Model\Entity;

use function array_map;
use function array_reduce;
use function file_get_contents;
use function filemtime;
use function is_file;
use function is_readable;
use function json_encode;
use function max;

use const JSON_THROW_ON_ERROR;

class MixinLookupCached implements MixinLookupInterface
{
    /**
     * @var MixinLookupInterface
     */
    private MixinLookupInterface $mixinLookup;

    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var Filesystem
     */
    private Filesystem $filesystem;

    /**
     * @var int|null
     */
    private ?int $modulesMTimeMax = null;

    /**
     * @param MixinLookupInterface   $mixinLookup
     * @param SourceMapInterface     $sourceMap
     * @param EntityFactoryInterface $entityFactory
     * @param Filesystem             $filesystem
     */
    public function __construct(
        MixinLookupInterface $mixinLookup,
        SourceMapInterface $sourceMap,
        EntityFactoryInterface $entityFactory,
        Filesystem $filesystem
    ) {
        $this->mixinLookup   = $mixinLookup;
        $this->sourceMap     = $sourceMap;
        $this->entityFactory = $entityFactory;
        $this->filesystem    = $filesystem;
    }

    /**
     * @param Entity $entity
     *
     * @return array|Entity[]
     * @throws EntityException
     * @throws LogicException
     */
    public function getMixins(Entity $entity): array
    {
        if (($mixins = $this->getSavedMixins($entity)) !== null) {
            return $mixins;
        }

        $mixins = $this->mixinLookup->getMixins($entity);

        $this->saveMetadata($entity, $mixins);

        return $mixins;
    }

    /**
     * @param Entity $entity
     *
     * @return array|null
     * @throws LogicException
     */
    public function getSavedMixins(Entity $entity): ?array
    {
        if (!is_file($entity->getSourcePath())) {
            return [];
        }

        if ($this->isEntityChanged($entity)) {
            return null;
        }

        $metadataMTime   = $this->getMetadataMTime($entity);
        $modulesMTimeMax = $this->getModulesMTimeMax();

        $modulesChanged = $modulesMTimeMax > $metadataMTime;

        if (
            !$modulesChanged
            && ($metadata = $this->loadMetadata($entity)) !== null
            && (!$metadata || $this->areExists($metadata))
        ) {
            return $metadata;
        }

        return null;
    }

    /**
     * @param Entity   $entity
     * @param Entity[] $mixins
     */
    protected function saveMetadata(Entity $entity, array $mixins): void
    {
        $metaFile = $this->getMetadataPath($entity);

        $mixinsFqn = array_map(static function (Entity $mixin) {
            return $mixin->getFqn();
        }, $mixins);

        try {
            $this->filesystem->dumpFile($metaFile, json_encode($mixinsFqn, JSON_THROW_ON_ERROR));
        } catch (JsonException $e) {
        }
    }

    /**
     * @param Entity $entity
     *
     * @return array|null
     * @throws LogicException
     */
    protected function loadMetadata(Entity $entity): ?array
    {
        $metaFile = $this->getMetadataPath($entity);

        if (!is_readable($metaFile)) {
            return null;
        }

        try {
            $mixinsFqn = json_decode(file_get_contents($metaFile), true, 512, JSON_THROW_ON_ERROR);
        } catch (JsonException $e) {
            return null;
        }

        $result = [];
        foreach ($mixinsFqn as $fqn) {
            $result[] = $this->entityFactory->build($fqn);
        }

        return $result;
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     */
    private function isEntityChanged(Entity $entity): bool
    {
        if (!is_file($entity->getTargetPath())) {
            return true;
        }

        return filemtime($entity->getSourcePath()) > filemtime($entity->getTargetPath());
    }

    /**
     * @param Entity $entity
     *
     * @return string
     */
    private function getMetadataPath(Entity $entity): string
    {
        return $entity->getTargetPath() . '.metadata';
    }

    /**
     * @param Entity $entity
     *
     * @return int
     */
    private function getMetadataMTime(Entity $entity): int
    {
        $metadataFile = $this->getMetadataPath($entity);

        return is_file($metadataFile) ? filemtime($metadataFile) : 0;
    }

    /**
     * @return int
     */
    private function getModulesMTimeMax(): int
    {
        if ($this->modulesMTimeMax === null) {
            $this->modulesMTimeMax = array_reduce($this->sourceMap->getModuleFiles(), static function ($carry, $file) {
                return max($carry, filemtime($file));
            }, 0);
        }

        return $this->modulesMTimeMax;
    }

    /**
     * @param Entity[] $data
     *
     * @return bool
     */
    private function areExists(array $data): bool
    {
        foreach ($data as $entity) {
            if (!is_readable($entity->getSourcePath())) {
                return false;
            }
        }

        return true;
    }
}
