<?php

declare(strict_types=1);

namespace XCartMarketplace\Connector;

use GuzzleHttp\Exception;
use GuzzleHttp\RequestOptions;
use JsonException;
use Psr\Http\Message\ResponseInterface;

class Sender
{
    /**
     * Send request and return Response object
     *
     * @param string $method Request method
     * @param string $url    Request URL
     * @param array  $params Request data
     *
     * @return Response
     */
    public static function send(string $method, string $url, array $params = [], array $headers = [], $timeout = 15, $connectTimeout = 5): Response
    {
        $sender = new \GuzzleHttp\Client();

        $response = new Response();

        $options = [];
        if ($params) {
            $options[RequestOptions::FORM_PARAMS] = $params;
        }

        if ($headers) {
            $options[RequestOptions::HEADERS] = $headers;
        }

        $options[RequestOptions::CONNECT_TIMEOUT] = $connectTimeout;
        $options[RequestOptions::TIMEOUT] = $timeout;

        try {

            $gResponse = $sender->request($method, $url, $options);

            if ($gResponse->getHeaderLine('content-type') === 'application/json') {
                $response->setResponse(json_decode((string) $gResponse->getBody(), true, 512, JSON_THROW_ON_ERROR));
            } else {
                $response->setResponse(static::prepareRawData($gResponse));
            }
        } catch (Exception\RequestException $e) {
            $response->setError('RequestException: ' . $e->getMessage());
        } catch (Exception\ConnectException $e) {
            $response->setError('ConnectException: ' . $e->getMessage());
        } catch (Exception\GuzzleException $e) {
            $response->setError('GuzzleException: ' . $e->getMessage());
        } catch (JsonException $e) {
            $response->setError('JsonException: ' . $e->getMessage());
        } catch (\Exception $e) {
            $response->setError('Exception: ' . $e->getMessage());
        }

        return $response;
    }

    /**
     * Format response raw data as an array
     *
     * @param ResponseInterface $gResponse Guzzle response
     * @return array
     */
    public static function prepareRawData(ResponseInterface $gResponse)
    {
        $headers = $gResponse->getHeaders();

        $contentRangeKey = isset($headers['Content-Range']) ? 'Content-Range' : 'content-range';
        if (isset($headers[$contentRangeKey])) {
            if (!preg_match('#[a-z\s]+(\d+)-(\d+)/(\d+)#i', $headers[$contentRangeKey][0], $matches)) {
                throw new \Exception('Unexpected content-range value: ' . $headers[$contentRangeKey]);
            }

            return [
                'body'  => $gResponse->getBody(),
                'from'  => $matches[1],
                'to'    => $matches[2],
                'total' => $matches[3],
            ];
        }

        return [
            'body' => $gResponse->getBody(),
        ];
    }
}
