<?php

declare(strict_types=1);

namespace XCartMarketplace\Connector\Request;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Constraints as Assert;
use XCartMarketplace\Connector\Validator;

abstract class ARequest
{
    protected const TTL_LONG  = 86400;
    protected const TTL_SHORT = 3600;

    /**
     * Request target
     *
     * @var string
     */
    protected string $target = '';

    /**
     * Request method
     *
     * @var string
     */
    protected string $method = 'POST';

    /**
     * Request cache TTL
     *
     * @var int
     */
    protected int $cacheTTL = self::TTL_SHORT;

    /**
     * Request timeout (guzzle)
     *
     * @var int
     */
    protected int $timeout;

    protected int $connectTimeout;

    /**
     * Request parameters
     *
     * @var array
     */
    protected array $params;

    /**
     * Flag: true to use cache to store response
     *
     * @var bool
     */
    protected bool $useCache = true;

    /**
     * Flag: true to ignore cached response for current request object
     *
     * @var bool
     */
    protected bool $ignoreCache = false;

    /**
     * Request's service options
     *
     * @var array
     */
    protected array $serviceOptions = [];

    /**
     * @param array $params Request parameters
     */
    public function __construct(array $params = [], bool $ignoreCache = false, array $serviceOptions = [], $timeout = 15, $connectTimeout = 15)
    {
        $this->params = $params;

        $this->ignoreCache = $ignoreCache;

        $this->serviceOptions = $serviceOptions;

        $this->timeout = $timeout;

        $this->connectTimeout = $connectTimeout;
    }

    /**
     * Validate config options agains the validation schema and return the list of errors
     *
     * @param Validator $validator Validator object
     *
     * @return string[]
     */
    public function validate(Validator $validator): array
    {
        return $validator->validate($this->params, static::getValidationSchema(), $this->getClass());
    }

    /**
     * Get request validation schema
     *
     * @return Constraint|null
     */
    public static function getValidationSchema(): ?Constraint
    {
        return null;
    }

    /**
     * Get response validation schema
     *
     * @return Constraint|null
     */
    public static function getResponseValidationSchema(): ?Constraint
    {
        return new Assert\Collection([
            new Assert\EqualTo('ok'),
        ]);
    }

    /**
     * Get response validation schema
     *
     * @return Constraint|null
     */
    public static function getErrorResponseValidationSchema(): ?Constraint
    {
        return new Assert\Collection([
            'error' => [
                new Assert\NotBlank(),
                new Assert\Positive(),
            ],
            'message' => [
                new Assert\NotBlank(),
                new Assert\Type(['type' => 'string']),
            ],
        ]);
    }

    /**
     * Format error response
     *
     * @param int    $code    Error code
     * @param string $message Error message
     * @return array
     */
    public function formatErrorResponse(int $code, string $message): array
    {
        return [
            'code' => $code,
            'message' => $message,
        ];
    }

    /**
     * Get request class name
     *
     * @return string
     */
    public function getClass(): string
    {
        return preg_replace('/.*\\\\(\w+)$/', '\\1', static::class);
    }

    /**
     * Get request cache key
     *
     * @param array $params
     * @return string
     */
    public function getCacheKey(array $params): string
    {
        return $this->target . '-' . md5(serialize($params));
    }

    /**
     * Return true if cached response should be ignored
     *
     * @return bool
     */
    public function isIgnoreCache(): bool
    {
        return !$this->useCache || $this->ignoreCache;
    }

    /**
     * Return true if cache is used to store response
     *
     * @return bool
     */
    public function isUseCache(): bool
    {
        return $this->useCache;
    }

    /**
     * Return true if request can be sent in get_dataset request
     * 
     * @return bool
     */
    public function isAllowedToSendInDataset(): bool
    {
        return in_array($this->target, GetDataset::getAllowedTargets());
    }

    /**
     * Get request target
     *
     * @return string
     */
    public function getTarget(): string
    {
        return $this->target;
    }

    /**
     * Get request method
     *
     * @return string
     */
    public function getMethod(): string
    {
        return $this->method;
    }

    /**
     * Get request cache TTL
     *
     * @return int
     */
    public function getCacheTTL(): int
    {
        return $this->cacheTTL;
    }

    /**
     * Get request timeout
     *
     * @return int
     */
    public function getTimeout(): int
    {
        return $this->timeout;
    }

    public function getConnectTimeout(): int
    {
        return $this->connectTimeout;
    }

    /**
     * Get request parameters
     *
     * @return array
     */
    public function getParams(): array
    {
        return $this->params;
    }

    /**
     * Get request headers
     *
     * @return array
     */
    public function getHeaders(): array
    {
        return [];
    }
}
