<?php

namespace App\Tests\Logic;

use App\Entity\CheckRequirementsResult;
use App\Exception\CircularDependencyException;
use App\Logic\CheckDependencyMap;
use App\Logic\CheckRequirements;
use MJS\TopSort\Implementations\StringSort;
use PHPUnit\Framework\TestCase;

class CheckRequirementsTest extends TestCase
{
    /**
     * @param array $expected
     * @param array $enabled
     * @param array $dependencyMap
     * @param array $actions
     *
     * @throws CircularDependencyException
     * @dataProvider providerCheckAction
     */
    public function testCheckAction(array $expected, array $enabled, array $sortedEnabledSkinModuleIds, array $dependencyMap, array $actions): void
    {
        $checkRequirements = new CheckRequirements(
            new CheckDependencyMap(
                new StringSort()
            )
        );

        $checkRequirements->setEnabledModuleIds($enabled);
        $checkRequirements->setSortedEnabledSkinModuleIds($sortedEnabledSkinModuleIds);
        $checkRequirements->setDependencyMap($dependencyMap);

        foreach ($actions as $action) {
            $checkRequirements->addAction(
                $action['actionType'],
                $action['moduleId'],
                $action['isSkin'],
                $action['canDisable'],
                '', // TODO add tests for minorRequiredCoreVersion
                $action['dependsOn'] ?? [],
            );
        }

        $result = $checkRequirements->checkActions();

        foreach ($expected as $propName => $propValue) {
            $getPropFunction = "get{$propName}";

            $this->assertEquals($propValue, $result->$getPropFunction());
        }
    }

    public function providerCheckAction(): array
    {
        return [
            'enable module'                                                         => [
                [
                    'toEnable'       => ['module1'],
                    'toDisable'      => [],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['module1' => 'enable'],
                    'failedActions'  => [],
                ],
                [],
                [],
                [],
                [
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'module1',
                        'isSkin'     => false,
                        'canDisable' => true,
                    ],
                ],
            ],
            'disable module'                                                        => [
                [
                    'toEnable'       => [],
                    'toDisable'      => ['module1'],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['module1' => 'disable'],
                    'failedActions'  => [],
                ],
                ['module1'],
                [],
                [],
                [
                    [
                        'actionType' => 'disable',
                        'moduleId'   => 'module1',
                        'isSkin'     => false,
                        'canDisable' => true,
                    ],
                ],
            ],
            'disable module and enable other module which has dependencies from it' => [
                [
                    'toEnable'       => [],
                    'toDisable'      => ['module1'],
                    'cantEnable'     => [
                        ['module2', CheckRequirementsResult::CANT_ENABLE_BY_DEPENDENCY_REASON, ['module1']],
                    ],
                    'cantDisable'    => [],
                    'successActions' => ['module1' => 'disable'],
                    'failedActions'  => ['module2' => 'enable'],
                ],
                ['module1'],
                [],
                [
                    'module2' => [
                        ['module1'],
                        [],
                    ],
                ],
                [
                    [
                        'actionType' => 'disable',
                        'moduleId'   => 'module1',
                        'isSkin'     => false,
                        'canDisable' => true,
                    ],
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'module2',
                        'isSkin'     => false,
                        'canDisable' => true,
                    ],
                ],
            ],
            'disable module when other enabled module has dependencies from it'     => [
                [
                    'toEnable'       => [],
                    'toDisable'      => [],
                    'cantEnable'     => [],
                    'cantDisable'    => [
                        ['module1', CheckRequirementsResult::CANT_DISABLE_BY_BACK_DEPENDENCY_REASON, ['module2']],
                    ],
                    'successActions' => [],
                    'failedActions'  => ['module1' => 'disable'],
                ],
                ['module1', 'module2'],
                [],
                [
                    'module2' => [
                        ['module1'],
                        [],
                    ],
                ],
                [
                    [
                        'actionType' => 'disable',
                        'moduleId'   => 'module1',
                        'isSkin'     => false,
                        'canDisable' => true,
                    ],
                ],
            ],

            'change skin'                                                           => [
                [
                    'toEnable'       => ['skinModule1'],
                    'toDisable'      => ['skinModule2'],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['skinModule1' => 'enable'],
                    'failedActions'  => [],
                ],
                ['skinModule2', 'module1', 'module2'],
                ['skinModule2'],
                [],
                [
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'skinModule1',
                        'isSkin'     => true,
                        'canDisable' => true,
                    ],
                ],
            ],
            'enable skin which is depend on other skin'                             => [
                [
                    'toEnable'       => ['skinModule2', 'skinModule1'],
                    'toDisable'      => ['skinModule3'],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['skinModule1' => 'enable'],
                    'failedActions'  => [],
                ],
                ['skinModule3', 'module1', 'module2'],
                ['skinModule3'],
                [
                    'skinModule1' => [
                        ['skinModule2'],
                        [],
                    ],
                ],
                [
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'skinModule1',
                        'isSkin'     => true,
                        'canDisable' => true,
                    ],
                ],
            ],
            'enable skin which is depend on already enabled skin'                   => [
                [
                    'toEnable'       => ['skinModule1'],
                    'toDisable'      => [],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['skinModule1' => 'enable'],
                    'failedActions'  => [],
                ],
                ['skinModule2', 'module1', 'module2'],
                ['skinModule2'],
                [
                    'skinModule1' => [
                        ['skinModule2'],
                        [],
                    ],
                ],
                [
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'skinModule1',
                        'isSkin'     => true,
                        'canDisable' => true,
                    ],
                ],
            ],
            'enable skin which was a dependency of already enabled skin'            => [
                [
                    'toEnable'       => [],
                    'toDisable'      => ['skinModule1'],
                    'cantEnable'     => [],
                    'cantDisable'    => [],
                    'successActions' => ['skinModule2' => 'enable'],
                    'failedActions'  => [],
                ],
                ['skinModule1', 'skinModule2', 'module1', 'module2'],
                ['skinModule1', 'skinModule2'],
                [
                    'skinModule1' => [
                        ['skinModule2'],
                        [],
                    ],
                ],
                [
                    [
                        'actionType' => 'enable',
                        'moduleId'   => 'skinModule2',
                        'isSkin'     => true,
                        'canDisable' => true,
                    ],
                ],
            ],
        ];
    }
}
