<?php

namespace App\Security\Http\Authenticator;

use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\Authenticator\AbstractAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Security\Core\User\InMemoryUser;

final class ApiKeyAuthenticator extends AbstractAuthenticator
{
    private Session $session;

    public function __construct(
        Session $session
    )
    {
        $this->session = $session;
    }

    public function supports(Request $request): bool
    {
        return (bool) $this->session;
    }

    public function authenticate(Request $request): Passport
    {
        $session   = $this->session;
        $profileId = $session->get('profile_id');

        if ($profileId) {
            return new SelfValidatingPassport(
                new UserBadge(
                    $profileId,
                    fn () => $this->getUserBySession($session)
                )
            );

        } else {
            throw new CustomUserMessageAuthenticationException('Access denied');
        }
    }

    public function getUserBySession(Session $session): ?UserInterface
    {
        if ($userId = $session->get('profile_id')) {
            $user = new InMemoryUser(
                'user_' . $userId,
                null,
                [
                    ($this->session->get('isAdmin') ? 'ROLE_ADMIN' : 'ROLE_USER')
                ]
            );
        }

        return $user ?? null;
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        return null;
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        $data = [
            'message' => strtr($exception->getMessageKey(), $exception->getMessageData())
        ];

        return new JsonResponse($data, Response::HTTP_UNAUTHORIZED);
    }
}
