<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Repository;

use App\Entity\Module;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Exception;
use JsonException;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use MJS\TopSort\Implementations\StringSort;

final class ModuleRepository
{
    private EntityManagerInterface $entityManager;

    private EntityRepository $repository;

    public function __construct(
        EntityManagerInterface $entityManager
    ) {
        $this->entityManager = $entityManager;
        $this->repository    = $entityManager->getRepository(Module::class);
    }

    /**
     * @param string $moduleId String with pair "{Author}-{Name}"
     */
    public function findByModuleId(string $moduleId): ?Module
    {
        return $this->repository->findOneBy(['moduleId' => $moduleId]);
    }

    /**
     * @param string $source
     *
     * @return mixed
     */
    public function deleteBySource(string $source)
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->delete(Module::class, 'm')
            ->where('m.source = :source')
            ->setParameter(':source', $source);

        return $qb->getQuery()->execute();
    }

    public function getSources(): array
    {
        $qb = $this->repository->createQueryBuilder('m');
        $qb
            ->select("m.moduleId moduleId")
            ->addSelect("CONCAT_WS('\\', m.author, m.name) namespace")
            ->addSelect("CONCAT_WS('/', m.author, m.name) path")
            ->where('m.source = :source')
            ->setParameter(':source', Module::SOURCE_LOCAL);

        $result = [];

        foreach ($qb->getQuery()->getArrayResult() as $data) {
            $result[$data['moduleId']] = [$data['namespace'], $data['path']];
        }

        return $result;
    }

    /**
     * @throws JsonException
     */
    public function getDependencyMap(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->addSelect("JSON_EXTRACT(m.metaData, '\$.dependsOn') dependsOn")
            ->addSelect("JSON_EXTRACT(m.metaData, '\$.incompatibleWith') incompatibleWith")
            ->where("JSON_CONTAINS_PATH(m.metaData, 'one', '\$.dependsOn[0]', '\$.incompatibleWith[0]') = 1");

        $result = [];

        foreach ($qb->getQuery()->getArrayResult() as $data) {
            $result[$data['moduleId']] = [
                json_decode($data['dependsOn'], true, 512, JSON_THROW_ON_ERROR),
                json_decode($data['incompatibleWith'], true, 512, JSON_THROW_ON_ERROR),
            ];
        }

        return $result;
    }

    /**
     * @return Module[]
     */
    public function findLocal(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb->where("m.source = :source")->setParameter(':source', Module::SOURCE_LOCAL);

        return $qb->getQuery()->getResult();
    }

    /**
     * @return array
     */
    public function getAllIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb->select("m.moduleId moduleId");

        return $qb->getQuery()->getSingleColumnResult();
    }

    /**
     * @return array
     */
    public function getEnabledIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->andWhere("m.state = :state")
            ->setParameter(':state', Module::STATE_ENABLED);

        return $qb->getQuery()->getSingleColumnResult();
    }

    /**
     * @return array
     */
    public function getDisabledIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->andWhere("m.state <> :state")
            ->setParameter(':state', Module::STATE_ENABLED);

        return $qb->getQuery()->getSingleColumnResult();
    }

    public function getEnabledModules(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select('m')
            ->where('m.state = :state')
            ->setParameter(':state', Module::STATE_ENABLED)
            ->orderBy('m.moduleId');

        return $qb->getQuery()->getResult();
    }

    public function getDisabledModules(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select('m')
            ->where('m.state <> :state')
            ->setParameter(':state', Module::STATE_ENABLED);

        return $qb->getQuery()->getResult();
    }

    public function getSortedEnabledIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->addSelect("JSON_EXTRACT(m.metaData, '\$.dependsOn') dependsOn")
            ->where('m.state = :state')
            ->setParameter(':state', Module::STATE_ENABLED);

        $sorter = new StringSort();
        foreach ($qb->getQuery()->getArrayResult() as $data) {
            $sorter->add(
                $data['moduleId'],
                json_decode($data['dependsOn'], true, 512, JSON_THROW_ON_ERROR)
            );
        }

        try {
            return $sorter->sort();
        } catch (CircularDependencyException $e) {
            // todo: throw domain exception
            throw $e;
        } catch (ElementNotFoundException $e) {
            // todo: throw domain exception
            throw $e;
        }
    }

    public function getEnabledSkinModuleIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->where('m.state = :state')
            ->andWhere("JSON_EXTRACT(m.metaData, '\$.type') = :type")
            ->setParameter(':state', Module::STATE_ENABLED)
            ->setParameter(':type', 'skin');

        return $qb->getQuery()->getSingleColumnResult();
    }

    /**
     * @throws CircularDependencyException
     * @throws ElementNotFoundException
     * @throws JsonException
     */
    public function getSortedEnabledSkinModuleIds(): array
    {
        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->addSelect("JSON_EXTRACT(m.metaData, '\$.dependsOn') dependsOn")
            ->where('m.state = :state')
            ->andWhere("JSON_EXTRACT(m.metaData, '\$.type') = :type")
            ->setParameter(':state', Module::STATE_ENABLED)
            ->setParameter(':type', 'skin');

        $enabledModules = [];
        foreach ($qb->getQuery()->getArrayResult() as $data) {
            $enabledModules[$data['moduleId']] = json_decode($data['dependsOn'], true, 512, JSON_THROW_ON_ERROR);
        }

        if (!$enabledModules) {
            return [];
        }

        $sorter = new StringSort();
        foreach ($enabledModules as $moduleId => $dependsOn) {
            $sorter->add($moduleId, array_filter($dependsOn, static function ($module) use ($enabledModules) {
                return isset($enabledModules[$module]);
            }));
        }

        try {
            return array_reverse($sorter->sort());
        } catch (CircularDependencyException $e) {
            // todo: throw domain exception
            throw $e;
        } catch (ElementNotFoundException $e) {
            // todo: throw domain exception
            throw $e;
        }
    }

    public function resetCustomFlag(): void
    {
        $qb = $this->repository->createQueryBuilder('m');
        $qb->update(Module::class, 'm');
        $qb->set('m.isCustom', 0);

        $qb->getQuery()->execute();
    }

    public function getAutoloaders(array $moduleIds): array
    {
        if (!$moduleIds) {
            return [];
        }

        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->select("m.moduleId moduleId")
            ->addSelect("CONCAT_WS('/', m.author, m.name) path")
            ->addSelect("JSON_EXTRACT(m.metaData, '\$.autoloader') autoloader")
            ->where($qb->expr()->in('m.moduleId', $moduleIds))
            ->andWhere("JSON_CONTAINS_PATH(m.metaData, 'one', '\$.autoloader[0]') = 1");

        return $qb->getQuery()->getArrayResult();
    }

    public function removeLocalModules(?array $modules = null)
    {
        // there is nothing to remove if empty array given
        if (is_array($modules) && !$modules) {
            return null;
        }

        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->delete(Module::class, 'm')
            ->where('m.source = :source')
            ->setParameter(':source', Module::SOURCE_LOCAL);

        // Remove all modules if null given
        if ($modules !== null) {
            $qb->andWhere($qb->expr()->in('m.moduleId', $modules));
        }

        return $qb->getQuery()->execute();
    }

    public function createModules(array $modules): void
    {
        if ($modules) {
            foreach ($modules as $moduleInfo) {
                $module = $this->createModuleFromArray($moduleInfo);

                $this->entityManager->persist($module);
            }

            $this->entityManager->flush();
        }
    }

    public function createModuleFromArray(array $moduleInfo): Module
    {
        $module = new Module();

        if (empty($moduleInfo['author']) || empty($moduleInfo['name'])) {
            // todo: use domain exception
            throw new Exception('Not enougth info for module');
        }

        $module->setModuleId($moduleInfo['author'] . '-' . $moduleInfo['name']);
        $module->setAuthor($moduleInfo['author']);
        $module->setName($moduleInfo['name']);
        $module->setVersion($moduleInfo['version'] ?? '');
        $module->setSource($moduleInfo['source'] ?? Module::SOURCE_LOCAL);
        $module->setState($moduleInfo['state'] ?? Module::STATE_NOT_INSTALLED);
        $module->setEnabledDate($moduleInfo['enabledDate'] ?? 0);
        $module->setMetaData($moduleInfo['metaData'] ?? []);
        $module->setHasLocalFiles($moduleInfo['hasLocalFiles'] ?? true);

        return $module;
    }

    public function removeModules(array $modules)
    {
        if (!$modules) {
            return;
        }

        $qb = $this->repository->createQueryBuilder('m');

        $qb
            ->delete(Module::class, 'm')
            ->where('m.moduleId IN (:ids)')
            ->setParameter(':ids', $modules);

        return $qb->getQuery()->execute();
    }

    /**
     * @return array|Module[]
     * @throws Exception
     */
    public function getAllModules(): array
    {
        return $this->repository->findAll();
    }

    public function removeAllFixtures(): void
    {
        array_map(
            static function (Module $module) {
                $metaData = $module->getMetaData();
                $metaData['fixtures'] = [];
                $module->setMetaData($metaData);
            },
            $this->getModulesWithNotEmptyFixtures()
        );
    }

    /**
     * @return Module[]
     */
    public function getModulesWithNotEmptyFixtures(): array
    {
        return $this->repository->createQueryBuilder('m')
            ->where('JSON_EXTRACT(m.metaData, :fixturesKey) != :emptyFixtures')
            ->setParameter('fixturesKey', '$.fixtures')
            ->setParameter('emptyFixtures', '[]')
            ->getQuery()
            ->execute();
    }
}
