<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Repository;

use App\Entity\License;
use App\Marketplace\Marketplace;
use App\Marketplace\MarketplaceStorage;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;

final class LicenseRepository
{
    private EntityRepository $repository;

    private EntityManagerInterface $entityManager;

    private MarketplaceStorage $storage;

    public function __construct(
        EntityManagerInterface $entityManager,
        MarketplaceStorage $storage
    ) {
        $this->repository    = $entityManager->getRepository(License::class);
        $this->entityManager = $entityManager;
        $this->storage       = $storage;
    }

    public function findCoreLicense(): ?License
    {
        return $this->repository->findOneBy([
            'moduleId' => Marketplace::CORE_MODULE_ID,
            'keyType'  => License::CORE_KEY_TYPE,
        ]);
    }

    public function findByModuleId(string $moduleId): ?License
    {
        return $this->repository->findOneBy([
            'moduleId' => $moduleId,
        ]);
    }

    public function findAll(): array
    {
        return $this->repository->findAll() ?? [];
    }

    public function createCoreLicense(): License
    {
        $coreLicense = new License();
        $coreLicense->setKeyValue(License::FREE_LICENSE_KEY);
        $coreLicense->setKeyType(License::CORE_KEY_TYPE);
        $coreLicense->setModuleId(Marketplace::CORE_MODULE_ID);
        $coreLicense->setExpiredAt($this->getXCartInstallationDate() + License::TRIAL_PERIOD);

        $this->entityManager->persist($coreLicense);
        $this->entityManager->flush();

        return $coreLicense;
    }

    public function getXCartInstallationDate(): int
    {
        return (int) ($this->storage->getValue('XCartInstallationDate') ?? (time() + License::TRIAL_PERIOD));
    }

    public function removeLicenseByKey(string $key): void
    {
        if ($license = $this->repository->findOneByKeyValue($key)) {
            $this->entityManager->remove($license);
            $this->entityManager->flush();
        }
    }

    public function getAddonLicenses(): array
    {
        return array_filter($this->findAll(), static function (License $license) {
            return $license->getKeyType() !== License::CORE_KEY_TYPE;
        });
    }

    public function getAddonLicenseKeyValues(): array
    {
        return array_reduce($this->getAddonLicenses(), static function (array $result, License $license): array {
            $result[$license->getModuleId()] = $license->getKeyValue();

            return $result;
        }, []);
    }
}
