<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\PrepareCode;

use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\Yaml\Yaml;

final class PrepareDistrCode implements PrepareCodeInterface
{
    private Filesystem $filesystem;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        Filesystem $filesystem,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->filesystem = $filesystem;
        $this->executeShellCommand = $executeShellCommand;
    }

    public function __invoke(PackerConfigInterface $packerConfig): void
    {
        $outputPath = $packerConfig->getOutputPath();
        $modules = array_merge(
            $packerConfig->getModules(),
            $packerConfig->getExternalModules(),
            $packerConfig->getDisabledModules()
        );
        $installConfiguration = $packerConfig->getInstallConfiguration();

        $this->installDependencies($outputPath);
        $this->cleanup($outputPath);
        $this->createEnvLocal($outputPath);
        $this->removeUnusedModules($outputPath, $modules);
        $this->saveInstallConfiguration($outputPath, $installConfiguration);
    }

    private function installDependencies(string $outputPath): void
    {
        ($this->executeShellCommand)('composer --no-interaction --no-scripts install', "{$outputPath}xcart/");
        ($this->executeShellCommand)('composer --no-interaction --no-scripts install', "{$outputPath}xcart/service-tool/");
    }

    private function cleanup(string $outputPath): void
    {
        $files = (new Finder())
            ->in("{$outputPath}xcart/")
            ->ignoreVCS(false)
            ->ignoreDotFiles(false)
            ->path($this->getPathsToDelete());

        $this->filesystem->remove(array_map(
            static fn (SplFileInfo $item) => $item->getPathname(),
            iterator_to_array($files)
        ));
    }

    private function createEnvLocal(string $outputPath): void
    {
        if ($this->filesystem->exists("{$outputPath}xcart/.env")) {

            $header = <<<HEADER
#                        !!! WARNING !!!
#
# DO NOT EDIT THE .env FILE. ANY CHANGES MADE TO THE .env FILE
# WILL BE REMOVED DURING THE NEXT STORE UPDATE. MANUALLY COPY .env TO .env.local
# (IF THERE IS NO SUCH FILE IN YOUR STORE YET) AND FILL IN THE REQUIRED VALUES
HEADER;
            $content = str_replace(
                $header,
                '',
                file_get_contents("{$outputPath}xcart/.env")
            );

            $this->filesystem->appendToFile("{$outputPath}xcart/.env.local", $content);
        }
    }

    private function getPathsToDelete(): array
    {
        return [
            '/\.git/',
            '.DS_Store',
            '.env.local',
            '.eslintrc.json',
            '.phpstorm.meta.php',
            'vendor/symfony/console/Resources/bin/hiddeninput.exe',
        ];
    }

    private function removeUnusedModules(string $outputPath, array $usedModules): void
    {
        $files = (new Finder())
            ->in("{$outputPath}xcart/modules")
            ->depth(1)
            ->directories()
            ->filter(static function (SplFileInfo $item) use ($usedModules) {
                $moduleId = str_replace('/', '-', $item->getRelativePathname());

                return !in_array($moduleId, $usedModules, true);
            });

        /** @var SplFileInfo $item */
        foreach (iterator_to_array($files) as $item) {
            $this->filesystem->remove($item->getPathname());
        }
    }

    private function saveInstallConfiguration(string $outputPath, array $installConfiguration): void
    {
        $this->filesystem->dumpFile("{$outputPath}xcart/var/install.yaml", Yaml::dump($installConfiguration));
    }
}
