<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\PrepareCode;

use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;

final class PrepareCoreCode implements PrepareCodeInterface
{
    private Filesystem $filesystem;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        Filesystem $filesystem,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->filesystem = $filesystem;
        $this->executeShellCommand = $executeShellCommand;
    }

    public function __invoke(PackerConfigInterface $packerConfig): void
    {
        $outputPath = $packerConfig->getOutputPath();

        if ($packerConfig->getSource() === 'git') {
            $this->installDependencies($outputPath);
        }

        $this->cleanup($outputPath);
    }

    private function installDependencies(string $outputPath): void
    {
        ($this->executeShellCommand)('composer --no-interaction --no-scripts install', "{$outputPath}xcart/");
        ($this->executeShellCommand)('composer --no-interaction --no-scripts install', "{$outputPath}xcart/service-tool/");
    }

    private function cleanup(string $outputPath): void
    {
        $files = (new Finder())
            ->in("{$outputPath}xcart/")
            ->ignoreDotFiles(false)
            ->path($this->getPathsToDelete());

        $this->filesystem->remove(array_map(
            static fn (SplFileInfo $item) => $item->getPathname(),
            iterator_to_array($files)
        ));
    }

    private function getPathsToDelete(): array
    {
        return [
            '/^\.DS_Store/',
            '/^\.env\.local/',
            '/^\.eslintrc\.json/',

            '.git',

            '/^\.phpstorm\.meta\.php/',

            '/^\.htaccess/',

            '/^CLOUDSEARCHTERMS.txt/',
            '/^LICENSE.txt/',

            '/^modules/',

            'node_modules',

            '/^public\/files/',
            '/^public\/images/',
            '/^public\/robots.txt/',

            '/^sql\/auto/',
            '/^sql\/demo/',
            '/^sql\/local/',
            '/^sql\/modules/',
            '/^sql\/site/',
            '/^sql\/ultimate/',

            'vendor/symfony/console/Resources/bin/hiddeninput.exe',
            '/^vendor\/symfony\/console\/Resources\/bin\/hiddeninput\.exe/',

            'vendor/laminas/laminas-code/docs',
            '/^vendor\/laminas\/laminas-code\/docs/',
        ];
    }
}
