<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\GetCode;

use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Finder\Finder;

final class GetLocalModulesCode implements GetCodeInterface
{
    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->executeShellCommand = $executeShellCommand;
    }

    public function __invoke(PackerConfigInterface $packerConfig): void
    {
        $sourcePath = $packerConfig->getSourcePath();
        $outputPath = $packerConfig->getOutputPath();
        $modules = $packerConfig->getModules();

        $modules = array_map(static fn ($item) => 'modules/' . str_replace('-', '/', $item), $modules);

        $excludes = $this->getExcludes($sourcePath, $modules);

        $modules = implode(' ', $modules);
        $excludes = implode(' ', $excludes);

        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand("-czf {$outputPath}xcart.tgz {$excludes} {$modules}"), $sourcePath);

        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand('-xzf xcart.tgz -C ./xcart'), $outputPath);

        ($this->executeShellCommand)('rm -rf xcart.tgz', $outputPath);
    }

    private function getExcludes(string $sourcePath, array $modules): array
    {
        $excludes = [];

        foreach($modules as $module) {
            $files =  (new Finder())->ignoreDotFiles(false)->name('/^\._/')->in($sourcePath . $module);

            $files = array_map(
                static fn (\SplFileInfo $file) => '--exclude=' . $file->getPathname(),
                iterator_to_array($files)
            );

            $excludes[] = $files;
        }

        return array_merge(...$excludes);
    }
}
