<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\GetCode;

use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Finder\Finder;

final class GetLocalCoreCode implements GetCodeInterface
{
    private ExecuteShellCommand $executeShellCommand;

    private GetCustomPaths $getCustomPaths;

    public function __construct(
        ExecuteShellCommand $executeShellCommand,
        GetCustomPaths $getCustomPaths
    ) {
        $this->executeShellCommand = $executeShellCommand;
        $this->getCustomPaths = $getCustomPaths;
    }

    public function __invoke(PackerConfigInterface $packerConfig): void
    {
        $sourcePath = $packerConfig->getSourcePath();
        $outputPath = $packerConfig->getOutputPath();

        $toExclude = $this->executeShellCommand->isBSDTar()
            ? array_map(static fn ($item) => "^{$item}", $this->getExcludePath($sourcePath))
            : $this->getExcludePath($sourcePath);

        $exclude = array_map(static fn ($item) => "--exclude=\"{$item}\"", $toExclude);
        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand(implode(' ', ["-czf {$outputPath}xcart.tgz", ...$exclude, './'])), $sourcePath);

        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand('-xzf xcart.tgz -C ./xcart'), $outputPath);

        ($this->executeShellCommand)('rm -rf xcart.tgz', $outputPath);
    }

    private function getExcludePath(string $sourcePath): array
    {
        $excludePaths = [
            './var',
            './service-tool/var',
            './public/assets',
            './public/bundles',
            './public/modules',
            './service-tool/public/bundles',
            './public/var',
            './.env.local',
            './.env.local.php',
            './.env.*.local',
            './config/secrets/prod/prod.decrypt.private.php',
            './config/dynamic',
            './config/local',
            './config/jwt',
        ];

        $excludeAssetsPaths = ($this->getCustomPaths)($sourcePath, 'assets');
        $excludeTemplatePaths = ($this->getCustomPaths)($sourcePath, 'templates');

        $dotFiles = (new Finder())->ignoreDotFiles(false)->name('/^\._/')->in($sourcePath);

        $dotFiles = array_map(
            static fn (\SplFileInfo $file) => \str_replace($sourcePath, './', $file->getPathname()),
            \iterator_to_array($dotFiles)
        );

        return \array_merge($excludePaths, $excludeAssetsPaths, $excludeTemplatePaths, \array_values($dotFiles));
    }
}
