<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\GetCode;

use App\Domain\XCart;
use App\Exception\DownloadModulesException;
use App\Exception\DownloadPackException;
use App\Marketplace\Marketplace;
use App\Operation\Build\ExtractPack;
use App\Output\XCartOutputInterface;
use XCartMarketplace\Connector\Client;
use XCartMarketplace\Connector\Config;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Contracts\Cache\CacheInterface;
use Psr\Log\LoggerInterface;
use stdClass;
use XCartMarketplace\Connector\Exceptions\ClientException;
use XCartMarketplace\Connector\Exceptions\RequestValidationException;

final class DownloadExternalModules
{
    private string $packsDir;

    private array $modules = [];

    private XCart $XCart;

    private Filesystem $filesystem;

    private ExtractPack $extractPack;

    private LoggerInterface $logger;

    private CacheInterface $cache;

    private Client $client;

    private XCartOutputInterface $output;

    private array $config;

    public function __construct(
        array $config,
        XCart $XCart,
        Filesystem $filesystem,
        ExtractPack $extractPack,
        LoggerInterface $logger,
        CacheInterface $cache,
        XCartOutputInterface $output
    ) {
        $this->XCart       = $XCart;
        $this->filesystem  = $filesystem;
        $this->extractPack = $extractPack;
        $this->logger      = $logger;
        $this->cache       = $cache;
        $this->config      = $config;
        $this->output      = $output;

        $this->packsDir = $XCart->getPacksPath();
    }

    /**
     * @throws DownloadPackException
     * @throws DownloadModulesException
     */
    public function __invoke(array $modulesToInstall = [], array $params = []): array
    {

        $params = array_merge($this->config, $params);
        $marketplaceConfig = new Config($this->prepareMarketplaceClientConfig((object) $params));
        $this->client      = new Client($marketplaceConfig, $this->cache, $this->logger);

        $modulesToInstall = $this->prepareModules($modulesToInstall);
        $packPaths = [];

        $this->filesystem->mkdir($this->packsDir);

        foreach ($modulesToInstall as $moduleId => $module) {
            $moduleVersion = Marketplace::implodeVersion($module['version']);
            $packagePath = "{$this->packsDir}{$moduleId}-v{$moduleVersion}";
            $upgradeSpec = array_merge($module, $params);

            $this->download($moduleId, $upgradeSpec, "{$packagePath}.tar.gz");
            ($this->extractPack)($packagePath);

            $packPaths[$moduleId] = $packagePath;
        }

        return $packPaths;
    }

    /**
     * @throws DownloadModulesException
     */
    private function prepareModules(array $modulesToInstall): array
    {
        $modules = [];
        $notActualVersionModules = [];

        preg_match('/^\d.\d/', $this->XCart->getCoreVersion(), $matches);
        $coreVersion = $matches[0];

        foreach ($modulesToInstall as $moduleId) {
            [$author, $name] = explode('-', $moduleId);
            $module = $this->getModule($author, $name);

            if (!$module) {
                throw DownloadModulesException::fromUnknownModule($moduleId);
            }

            $moduleVersion = Marketplace::implodeVersion($module['version']);
            if (strpos($moduleVersion, $coreVersion) !== 0) {
                $notActualVersionModules[] = $moduleId;
                continue;
            }

            $modules[$moduleId] = $module;
        }

        if ($notActualVersionModules) {
            throw DownloadModulesException::fromNotActualVersionModules($notActualVersionModules);
        }

        return $modules;
    }

    private function getModule(string $author, string $name): array
    {
        if (!$this->modules) {
            $this->client->addRequest('get_addons');
            $this->modules = $this->client->getData()['get_addons']['modules'] ?? [];
        }

        foreach ($this->modules as $module) {
            if (
                $author === $module['author']
                && $name === $module['name']
            ) {
                return $module;
            }
        }

        return [];
    }

    /**
     * @throws DownloadPackException
     */
    private function download(string $moduleId, array $upgradeSpec, string $targetPath): void
    {
        if ($this->filesystem->exists($targetPath)) {
            $this->filesystem->remove($targetPath);
        }

        try {
            $this->output->addInitMessage("Download package {$moduleId}");

            $this->downloadModule($moduleId, $upgradeSpec, $targetPath);

            $this->output->addEndMessage("[OK]");
            $this->output->addEndMessage("The package is successfully stored at {$targetPath}");
        } catch (DownloadPackException $e) {
            $this->output->addErrorMessage($e->getMessage());

            throw $e;
        }
    }

    /**
     * @throws ClientException
     * @throws RequestValidationException
     * @throws DownloadPackException
     */
    private function downloadModule(string $moduleId, array $upgradeSpec, string $targetPath): void
    {
        $fp = fopen($targetPath, 'a+');

        $this->client->addRequest('get_addon_pack', ['moduleId' => $upgradeSpec['moduleId'], 'gzipped' => 1]);
        $response = $this->client->getData();

        if (isset($response['error'])) {
            throw DownloadPackException::fromResponseWithError($response['message'], $moduleId);
        }

        $data = $response['get_addon_pack']['body'] ?? null;

        if ($data) {
            fwrite($fp, (string) $data);
        } else {
            throw DownloadPackException::fromEmptyResponse($moduleId);
        }
    }

    private function prepareMarketplaceClientConfig(stdClass $config): array
    {
        return [
            'url'                => $config->url,
            'shopID'             => 'localhost',
            'shopDomain'         => 'localhost',
            'shopURL'            => 'http://localhost',
            'currentCoreVersion' => Marketplace::explodeVersion($this->XCart->getCoreVersion()),
            'email'              => 'admin@example.com',
            'installation_lng'   => $config->installationLng,
            'shopCountryCode'    => 'US',
            'affiliateId'        => $config->affiliateId,
            'trial'              => $config->isTrial,
            'cloud'              => false,
            'xcn_license_key'    => $config->xcn_license_key ?? '',
        ];
    }
}
