<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\CreatePack;

use App\Domain\ModuleDomain;
use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;

final class CreateModulesPack implements CreatePackInterface
{
    private ModuleDomain $moduleDomain;

    private Filesystem $filesystem;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        ModuleDomain $moduleDomain,
        Filesystem $filesystem,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->moduleDomain        = $moduleDomain;
        $this->filesystem          = $filesystem;
        $this->executeShellCommand = $executeShellCommand;
    }

    public function __invoke(PackerConfigInterface $packerConfig): array
    {

        $outputPath = $packerConfig->getOutputPath();
        $modules    = $packerConfig->getModules();

        $this->moduleDomain->setSourcePath("{$outputPath}xcart/");

        $result = [];

        foreach ($modules as $moduleId) {
            $data = $this->moduleDomain->readModuleInfo($moduleId);

            $this->createMetaFile($outputPath, $data);
            $this->createHash($outputPath, $moduleId);

            $result[] = $this->pack($outputPath, $moduleId, $data['version']);
        }

        return $result;
    }

    private function createMetaFile(string $outputPath, array $data): void
    {
        [$majorVersion1, $majorVersion2, $minorVersion, $buildVersion] = explode('.', $data['version']);

        $metaData = [
            'RevisionDate'      => time(),
            'ActualName'        => $data['author'] . '\\' . $data['name'],
            'VersionMajor'      => $majorVersion1 . '.' . $majorVersion2,
            'VersionMinor'      => $minorVersion,
            'VersionBuild'      => $buildVersion,
            'MinCoreVersion'    => $data['metaData']['minorRequiredCoreVersion'],
            'Name'              => $data['metaData']['moduleName'],
            'Author'            => $data['metaData']['authorName'],
            'Description'       => $data['metaData']['description'],
            'isSystem'          => $data['metaData']['isSystem'],
            'Dependencies'      => array_map(
                static fn ($value) => str_replace('-', '\\', $value),
                $data['metaData']['dependsOn'] ?? []
            ),
            'Incompatibilities' => array_map(
                static fn ($value) => str_replace('-', '\\', $value),
                $data['metaData']['incompatibleWith'] ?? []
            ),
        ];

        $this->filesystem->dumpFile(
            "{$outputPath}xcart/.phar/.metadata.bin",
            serialize($metaData)
        );
    }

    private function createHash(string $outputPath, string $moduleId): void
    {
        $hash = [];

        $files = (new Finder())
            ->in("{$outputPath}xcart/modules/" . str_replace('-', '/', $moduleId))
            ->ignoreDotFiles(false)
            ->files();

        $files = array_map(
            static fn (SplFileInfo $item) => str_replace("{$outputPath}xcart/", '', $item->getPathname()),
            iterator_to_array($files)
        );

        foreach ($files as $fullPath => $file) {
            $hash[$file] = md5_file($fullPath);
        }

        $this->filesystem->dumpFile(
            "{$outputPath}xcart/.hash",
            json_encode($hash, JSON_THROW_ON_ERROR)
        );
    }

    private function pack(string $outputPath, string $moduleId, string $version): string
    {
        $archive = "{$outputPath}{$moduleId}-v{$version}.tar.gz";

        $arguments = '--numeric-owner '
            . ($this->executeShellCommand->isBSDTar() ? '--uid=0 --gid=0 ' : '--owner=0 --group=0 ')
            . "-czf {$archive} "
            . 'modules/' . str_replace('-', '/', $moduleId) . '/* '
            . '.phar '
            . '.hash ';

        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand($arguments), "{$outputPath}xcart/");

        return $archive;
    }
}
