<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Pack\CreatePack;

use App\Marketplace\Marketplace;
use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfigInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;

final class CreateCorePack implements CreatePackInterface
{
    use VersionReplaceTrait;

    private Filesystem $filesystem;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        Filesystem $filesystem,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->filesystem = $filesystem;
        $this->executeShellCommand = $executeShellCommand;
    }

    public function __invoke(PackerConfigInterface $packerConfig): array
    {
        $outputPath = $packerConfig->getOutputPath();
        $version = $packerConfig->getVersion();

        $this->createMetaFile($outputPath, $version);
        $this->setVersion($outputPath, $version);
        $this->createHash($outputPath);

        return [$this->pack($outputPath, $version)];
    }

    private function createMetaFile(string $outputPath, string $version): void
    {
        [$majorVersion1, $majorVersion2, $minorVersion, $buildVersion] = explode('.', $version);

        $metaData = [
            'RevisionDate' => time(),
            'VersionMajor' => $majorVersion1 . '.' . $majorVersion2,
            'VersionMinor' => $minorVersion,
            'VersionBuild' => $buildVersion,
        ];

        $this->filesystem->dumpFile(
            "{$outputPath}xcart/.phar/.metadata.bin",
            serialize($metaData)
        );
    }

    private function createHash(string $outputPath): void
    {
        $hash = [];

        $files = (new Finder())
            ->in("{$outputPath}xcart/")
            ->ignoreDotFiles(false)
            ->files();

        $files = array_map(
            static fn (SplFileInfo $item) => str_replace("{$outputPath}xcart/", '', $item->getPathname()),
            iterator_to_array($files)
        );

        foreach ($files as $fullPath => $file) {
            $hash[$file] = md5_file($fullPath);
        }

        $this->filesystem->dumpFile(
            "{$outputPath}xcart/.hash",
            json_encode($hash, JSON_THROW_ON_ERROR)
        );
    }

    private function pack(string $outputPath, string $version): string
    {
        $coreId = Marketplace::CORE_MODULE_ID;
        $archive = "{$outputPath}{$coreId}-v{$version}.tar.gz";

        ($this->executeShellCommand)($this->executeShellCommand->buildTarCommand("-czf {$archive} ."), "{$outputPath}xcart/");

        return $archive;
    }
}
