<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Integrity;

use App\Domain\XCart;
use App\Exception\GetHashException;
use App\Marketplace\Marketplace;
use App\Marketplace\MarketplaceClient;
use App\Repository\ModuleRepository;

final class ModuleHashRetriever
{
    private XCart $XCart;

    private MarketplaceClient $client;

    private ModuleRepository $moduleRepository;

    public function __construct(
        XCart             $XCart,
        MarketplaceClient $client,
        ModuleRepository  $moduleRepository
    ) {
        $this->XCart            = $XCart;
        $this->client           = $client;
        $this->moduleRepository = $moduleRepository;
    }

    /**
     * @throws GetHashException
     */
    public function __invoke(array $moduleVersionHashes): array
    {
        if (!$moduleVersionHashes) {
            return [];
        }

        $rootPath = $this->XCart->getSourcePath();

        $paths       = [];
        $modulesData = [];

        $modulesSources = $this->moduleRepository->getSources();
        foreach ($modulesSources as $moduleId => [$namespace, $source]) {
            if (isset($moduleVersionHashes[$moduleId])) {
                $paths[$moduleId] = $rootPath . 'modules/' . $source;
            }
        }

        $hashes = $this->client->retrieve(
            'get_addon_hash_batch',
            [
                'moduleId' => array_values($moduleVersionHashes),
            ]
        );

        if (isset($hashes['error']) || isset($hashes['code'])) {
            throw GetHashException::fromGetModuleHashResponseWithError($hashes['message']);
        }

        $hashesId = array_flip($moduleVersionHashes);
        foreach ($hashes as $key => $hash) {
            if (!isset($hash['error'])) {
                $moduleId = $hashesId[$key] ?? null;
                if ($moduleId) {
                    try {
                        $hashResult             = json_decode($hash['result'], true, 512, JSON_THROW_ON_ERROR);
                        $modulesData[$moduleId] = [
                            'result'      => $hashResult,
                            'path'        => $paths[$moduleId],
                            'versionHash' => $key,
                        ];
                    } catch (\JsonException $e) {
                        $modulesData[$moduleId] = [
                            'error' => $e->getMessage(),
                        ];
                    }
                }
            }
        }

        ksort($modulesData);

        return $modulesData;
    }
}
