<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Integrity;

use App\Domain\XCart;

final class GenerateViolations
{
    public function __construct(
        private XCart $xcart
    ) {
    }

    public function __invoke(array $known, array $actual)
    {
        $entries = $this->buildFilesystemEntries($known, $actual);

        return $this->getViolations($entries);
    }

    /**
     * @param array $knownHashes
     * @param array $actualFiles
     *
     * @return array
     */
    protected function buildFilesystemEntries($knownHashes, $actualFiles)
    {
        $result = [];

        foreach ($knownHashes as $path => $knownHash) {
            $result[$path] = [
                'known'  => $knownHash,
                'actual' => null
            ];
        }

        foreach ($actualFiles as $path => $entry) {
            if (!isset($result[$path])) {
                $result[$path] = [
                    'known'  => null,
                    'actual' => null
                ];
            }

            $result[$path]['actual'] = $entry;
        }

        return $result;
    }

    /**
     * @param array $filesystemEntries
     *
     * @return array
     */
    public function getViolations($filesystemEntries): array
    {
        $result = [];

        $filesystemEntries = static::postProcessFiles($filesystemEntries);
        foreach ($filesystemEntries as $path => $filesystemEntry) {
            $actual      = $filesystemEntry['actual'];
            $known       = $filesystemEntry['known'];

            if ($known && substr($known, 0, 1) === '/') {
                $known = file_exists($known)
                    ? md5_file($known)
                    : '';
            }

            if ($actual) {
                $file = substr($actual, 0, 1) === '/'
                    ? $actual
                    : $this->xcart->getSourcePath() . $path;

                $actual = file_exists($file)
                    ? md5_file($file)
                    : null;
            }

            $resultEntry = [
                'filepath'    => $path,
                'hash_actual' => '',
                'hash_known'  => (string) $known,
                'type'        => 'unknown',
            ];

            if ($actual && !$known) {
                $resultEntry['type'] = 'added';
                $result[]            = $resultEntry;
                continue;
            }

            if (!$actual && $known) {
                $resultEntry['type'] = 'removed';
                $result[]            = $resultEntry;
                continue;
            }

            if ($actual !== $known) {
                $resultEntry['hash_actual'] = $actual;
                $resultEntry['type']        = 'modified';
                $result[]                   = $resultEntry;
            }
        }

        return $result;
    }

    /**
     * @return string
     */
    private static function getExcludedPattern(): string
    {
        $list = array_merge(
            [
                'list' => [],
                'raw'  => [],
            ],
            static::getCommonExcludePatterns()
        );

        $toImplode = $list['raw'];

        foreach ($list['list'] as $pattern) {
            $toImplode[] = preg_quote($pattern, '/');
        }

        return '/^(?:' . implode('|', $toImplode) . ')/Ss';
    }

    /**
     * @return array
     */
    private static function getCommonExcludePatterns(): array
    {
        $patterns = [
            ".*\/\.gitattributes",
            ".*\/\.gitignore",
            ".*\/?\.htaccess",
            ".*\/?\.DS_Store",
            ".*\/?\.Modules.php",
            ".*\.log$",
            ".*skins\/common\/images\/flags_svg\/.*\.svg",
            "\.phar\/\.metadata\.bin",
            "phpunit\.xml",
            ".*\.phpstorm\.meta\.php",
            ".*\/composer\/autoload_psr4\.php",
            ".*\/composer\/installed\.php",
        ];

        return [
            'list' => [],
            'raw'  => $patterns,
        ];
    }

    /**
     * @param array $files
     *
     * @return array
     */
    private static function postProcessFiles(array $files): array
    {
        $pattern = static::getExcludedPattern();

        return array_filter($files, static function ($file) use ($pattern) {
            return !preg_match($pattern, (string) $file);
        }, ARRAY_FILTER_USE_KEY);
    }
}
