<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Integrity;

use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;

final class ActualFilesRetriever
{
    private Filesystem $filesystem;

    public function __construct(Filesystem $filesystem) {
        $this->filesystem = $filesystem;
    }

    public function __invoke(string $path, string $moduleId = ''): array
    {
        return $this->getActualFilesPaths($path, $moduleId);
    }

    protected function getActualFilesPaths(string $path, string $moduleId): array
    {
        // for the delayed upgrade the directory $path does not exist
        if (!$this->filesystem->exists($path)) {
            return [];
        }

        $actual = [];

        $files = $moduleId ? $this->getDirectoryContent($path) : $this->getCoreContent($path);
        /** @var \Symfony\Component\Finder\SplFileInfo $file */
        foreach ($files as $file) {
            $relative = $moduleId
                ? ('modules/' . str_replace('-', '/', $moduleId) . '/' . $file->getRelativePathname())
                : $file->getRelativePathname();

            $actual[$relative] = $file->getPathname();
        }

        return $actual;
    }

    protected function getCoreContent(string $directory): array
    {
        $excluded = [
            '/^\.DS_Store/',
            '/^\.env\.local/',
            '/^\.env\.local\.php/',
            '/^\.eslintrc\.json/',

            '.git',
            '.gitignore',
            '.gitkeep',

            '/^\.htaccess/',

            '/^\.idea/',

            '/^\.phpstorm\.meta\.php/',
            '/^\.phar\/\.metadata\.bin/',

            '/^LICENSE\.txt/',
            '/^CLOUDSEARCHTERMS\.txt/',

            '/^config\/jwt/',
            '/^config\/local/',
            '/^config\/dynamic/',
            '/^config\/secrets\/prod\/prod\.decrypt\.private\.php/',

            '/^modules/',

            'node_modules',

            '/^public\/assets/',
            '/^public\/bundles/',
            '/^public\/files/',
            '/^public\/images/',
            '/^public\/modules/',
            '/^public\/robots.txt/',
            '/^public\/var/',

            '/^assets\/custom_web/',
            '/^assets\/custom_mail/',
            '/^assets\/custom_pdf/',
            '/^templates\/custom_web/',
            '/^templates\/custom_mail/',
            '/^templates\/custom_pdf/',

            '/^service-tool\/config\/local/',
            '/^service-tool\/public\/bundles/',
            '/^service-tool\/var/',

            '/^sql\/auto/',
            '/^sql\/demo/',
            '/^sql\/local/',
            '/^sql\/modules/',
            '/^sql\/site/',
            '/^sql\/ultimate/',

            '/^var\//',

            '/^vendor\/\.htaccess/',
            '/^vendor\/autoload_xcart\.php/',
            '/vendor\/symfony\/console\/Resources\/bin\/hiddeninput\.exe/',
        ];

        return $this->getDirectoryContent($directory, $excluded);
    }

    protected function getDirectoryContent(string $directory, array $exclude = []): array
    {
        $finder = Finder::create()
            ->ignoreVCS(true)
            ->ignoreDotFiles(false)
            ->in($directory)
            ->notPath($exclude);

        return iterator_to_array($finder->files(), true);
    }
}
