<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation;

use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\ExecutableFinder;
use Symfony\Component\Process\PhpExecutableFinder;
use Symfony\Component\Process\Process;

final class ExecuteShellCommand
{
    private string $tarExecutable;

    private string $phpBinaryPath;

    public function __construct(
        string $tarExecutable
    ) {
        if (empty($tarExecutable)) {
            $finder = new ExecutableFinder();

            $tarExecutable = $finder->find('tar') ?: '';
        }

        $this->tarExecutable = $tarExecutable;

        $this->phpBinaryPath = (new PhpExecutableFinder())->find() ?: 'php';
    }

    /**
     * @param array|string $command
     */
    public function __invoke($command, string $cwd = null): string
    {
        if (is_array($command)) {
            $process = new Process($this->prepareCommand($command), $cwd);
        } else {
            $process = Process::fromShellCommandline($this->prepareCommand($command), $cwd);
        }


        $process->setTimeout(null);

        try {
            $process->run();

            if ($process->isSuccessful()) {
                return $process->getOutput();
            }

            throw new ProcessFailedException($process);

        } catch (\Exception $e) {
            throw new ProcessFailedException($process);
        }
    }

    public function buildTarCommand(string $arguments): string
    {
        return $this->tarExecutable . ' ' . $arguments;
    }

    public function isBSDTar(): bool
    {
        $version = ($this)("{$this->tarExecutable} --version");

        return strpos($version, 'bsdtar') !== false;
    }

    /**
     * @param array|string $command
     *
     * @return array|string
     */
    private function prepareCommand($command)
    {
        if (is_string($command) && strpos($command, 'php ') === 0) {
            $command = substr_replace($command, $this->phpBinaryPath, 0, 3);
        }

        if (is_array($command) && $command[0] === 'php') {
            $command[0] = $this->phpBinaryPath;
        }

        return $command;
    }
}
