<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Download;

use App\Domain\XCart;
use App\Repository\LicenseRepository;
use Symfony\Component\Filesystem\Filesystem;

final class MarketplaceDownloader extends AbstractDownloader
{
    private int $chunkSize;

    private string $packsDir;

    private Filesystem $filesystem;

    public function __construct(
        int $chunkSize,
        LicenseRepository $licenseRepository,
        ChunkRetrieverInterface $chunkRetriever,
        Filesystem $filesystem,
        XCart $XCart
    ) {
        parent::__construct($licenseRepository, $chunkRetriever);

        $this->chunkSize  = $chunkSize;
        $this->filesystem = $filesystem;
        $this->packsDir   = $XCart->getPacksPath();
    }

    public function download(
        string $moduleId,
        array $moduleUpgradeData,
        string $targetPath,
        string $target = '',
        array $params = [],
        int $current = 0,
        int $totalSize = 0
    ): void {
        if ($current === 0) {
            if ($this->filesystem->exists($targetPath)) {
                $this->filesystem->remove($targetPath);
            }

            if (!$this->filesystem->exists($this->packsDir)) {
                $this->filesystem->mkdir($this->packsDir);
            }
        }

        $requestTarget = $target ?: $this->getTarget($moduleId);
        $requestParams = $params ?: $this->getParams($moduleId, $moduleUpgradeData);

        $this->downloadPack($requestTarget, $requestParams, $targetPath, $moduleId, $current, $totalSize);
    }

    private function downloadPack(
        string $target,
        array $params,
        string $targetPath,
        string $moduleId,
        int $current = 0,
        int $totalSize = 0
    ): void {
        $size = $totalSize ?: $this->getSize($target, $params, $moduleId);
        $fp   = fopen($targetPath, 'a+');

        while ($current < $size) {
            $this->chunkRetriever->retrieve(
                $fp,
                $target,
                $params,
                $current . '-' . (min($current + $this->chunkSize, $size) - 1),
                $moduleId
            );

            $current += $this->chunkSize;
        }
    }
}
