<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Download;

use App\Marketplace\Marketplace;
use App\Repository\LicenseRepository;

abstract class AbstractDownloader implements DownloaderInterface
{
    private LicenseRepository $licenseRepository;

    protected ChunkRetrieverInterface $chunkRetriever;

    public function __construct(
        LicenseRepository $licenseRepository,
        ChunkRetrieverInterface $chunkRetriever
    ) {
        $this->licenseRepository = $licenseRepository;
        $this->chunkRetriever    = $chunkRetriever;
    }

    public function getSize(string $target, array $params, string $moduleId): int
    {
        return $this->chunkRetriever->retrieveSize($target, $params, $moduleId);
    }

    public function getTarget(string $moduleId): string
    {
        return ($moduleId === Marketplace::CORE_MODULE_ID)
            ? 'get_core_pack'
            : 'get_addon_pack';
    }

    public function getParams(string $moduleId, array $moduleUpgradeData): array
    {
        $params = ['gzipped' => 1];

        if ($moduleId === Marketplace::CORE_MODULE_ID) {
            $params['version'] = $moduleUpgradeData['version'];
        } else {
            $license = $this->licenseRepository->findByModuleId($moduleId);

            $params['moduleId'] = $moduleUpgradeData['moduleId']; // module version hash
            $params['key']      = $license ? $license->getKeyValue() : null;
        }

        if (isset($moduleUpgradeData['xcn_license_key'])) {
            $params['xcn_license_key'] = $moduleUpgradeData['xcn_license_key'];
        }

        return $params;
    }
}
