<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Upgrade54To55;

use App\Domain\XCart;
use App\Entity\License;
use App\Entity\Module;
use App\Marketplace\MarketplaceStorage;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Yaml\Yaml;

final class MoveServiceToolData
{
    private string $serviceToolDataPath;

    private string $modulesPath;

    private string $tablePrefix;

    private ModuleRepository $moduleRepository;

    private EntityManagerInterface $entityManager;

    private MarketplaceStorage $storage;

    private Filesystem $filesystem;

    public function __construct(
        string $tablePrefix,
        XCart $XCart,
        ModuleRepository $moduleRepository,
        EntityManagerInterface $entityManager,
        MarketplaceStorage $storage,
        Filesystem $filesystem
    ) {
        $this->tablePrefix         = $tablePrefix;
        $this->modulesPath         = $XCart->getModulesPath();
        $this->serviceToolDataPath = "{$XCart->getSourcePath()}files/service/";
        $this->moduleRepository    = $moduleRepository;
        $this->entityManager       = $entityManager;
        $this->storage             = $storage;
        $this->filesystem          = $filesystem;
    }

    /**
     * @throws \Exception
     */
    public function __invoke(): void
    {
        $this->moveModules();
        $this->moveLicenses();
        $this->moveCoreData();
        $this->moveDisabledStructures();

        $this->entityManager->flush();
    }

    /**
     * @throws \Exception
     */
    private function moveModules(): void
    {
        $installedModules = $this->getUnserializedDataFromFile('busInstalledModulesStorage');

        foreach ($installedModules as $module) {
            $module   = (array) $module;
            $moduleId = $module['id'];

            if (in_array($moduleId, ['CDev-Core', 'XC-Service', 'XC-Cloud'], true)) {
                continue;
            }

            $module['enabledDate'] = (int) ($module['enabledDate'] ?? 0);

            $path = $this->modulesPath . str_replace('-', '/', $moduleId);
            if (!$this->filesystem->exists($path)) {
                $module['hasLocalFiles'] = false;
            }

            if ($module['installed']) {
                $module['state'] = $module['enabled']
                    ? Module::STATE_ENABLED
                    : Module::STATE_INSTALLED;
            } else {
                $module['state'] = Module::STATE_NOT_INSTALLED;
            }

            $module['metaData'] = [
                'type'                     => $module['type'] ?? 'common',
                'isSystem'                 => $module['isSystem'] ?? false,
                'dependsOn'                => $module['dependsOn'] ?? [],
                'authorName'               => $module['authorName'] ?? '',
                'canDisable'               => $module['canDisable'] ?? true,
                'moduleName'               => $module['moduleName'] ?? '',
                'description'              => $module['description'] ?? '',
                'incompatibleWith'         => $module['incompatibleWith'] ?? [],
                'showSettingsForm'         => $module['showSettingsForm'] ?? false,
                'minorRequiredCoreVersion' => $module['minorRequiredCoreVersion'] ?? '0',
            ];

            $moduleEntity = $this->moduleRepository->createModuleFromArray($module);

            $this->entityManager->persist($moduleEntity);
        }
    }

    private function moveLicenses(): void
    {
        $licenses = $this->getUnserializedDataFromFile('licenseStorage');

        foreach ($licenses as $license) {
            $name   = $license['name'] ?? '';
            $author = $license['author'] ?? '';

            if ($name && $author) {
                $moduleId = "{$author}-{$name}";

                $licenseEntity = new License();
                $licenseEntity->setModuleId($moduleId);
                $licenseEntity->setKeyType((int) ($license['keyType'] ?? ''));
                $licenseEntity->setKeyValue($license['keyValue'] ?? '');
                $licenseEntity->setKeyData($license['keyData'] ?? []);
                $licenseEntity->setExpiredAt($license['keyData']['expDate'] ?? 0);
                $licenseEntity->setRegisteredAt();
                $licenseEntity->setUpdatedAt();

                $this->entityManager->persist($licenseEntity);
            }
        }
    }

    private function moveDisabledStructures(): void
    {
        $filePath = "{$this->serviceToolDataPath}.disabled.structures.php";

        if ($this->filesystem->exists($filePath)) {
            $disabledStructures = Yaml::parseFile($filePath);

            foreach ($disabledStructures as $moduleId => $structure) {
                $module = $this->moduleRepository->findByModuleId($moduleId);

                if ($module) {
                    $module->setStructure(
                        $this->getStructureWithTablePrefix($structure)
                    );
                }
            }
        }
    }

    private function getStructureWithTablePrefix(array $structure): array
    {
        $newStructure = [
            'tables'       => [],
            'columns'      => [],
            'dependencies' => [],
        ];

        foreach ($structure['tables'] as $table) {
            $newStructure['tables'][] = "{$this->tablePrefix }_{$table}";
        }

        foreach ($structure['columns'] as $table => $columns) {
            $newStructure['columns']["{$this->tablePrefix }_{$table}"] = $columns;
        }

        foreach ($structure['dependencies'] as $module => $tables) {
            $module = str_replace('-', '\\', $module);

            foreach ($tables as $table => $columns) {
                $newStructure['dependencies'][$module]["{$this->tablePrefix }_{$table}"] = $columns;
            }
        }

        return $newStructure;
    }

    private function moveCoreData(): void
    {
        $coreData = $this->getUnserializedDataFromFile('coreConfigStorage');

        if (isset($coreData['version'])) {
            $this->storage->setValue(
                'XCartCoreVersion',
                $coreData['version']
            );
        }

        if (isset($coreData['dataDate'])) {
            $this->storage->setValue(
                'XCartInstallationDate',
                (string) $coreData['dataDate']
            );
        }
    }

    private function getUnserializedDataFromFile(string $fileName): array
    {
        $fileContent = file_get_contents("{$this->serviceToolDataPath}{$fileName}.data");

        return unserialize(
            $fileContent,
            ['allowed_classes' => true]
        );
    }
}
