<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Upgrade;

use App\Domain\XCart;
use App\Exception\GetHashException;
use App\Marketplace\Marketplace;
use App\Operation\Build\ExtractPack;
use App\Operation\Integrity\ActualFilesRetriever;
use App\Operation\Integrity\CoreHashRetriever;
use App\Operation\Integrity\GenerateViolations;
use App\Operation\Integrity\ModuleHashRetriever;
use App\Repository\ModuleRepository;

final class GenerateUpgradeEntries
{
    private XCart $XCart;

    private CoreHashRetriever $coreHashRetriever;

    private ModuleHashRetriever $moduleHashRetriever;

    private ActualFilesRetriever $actualFilesRetriever;

    private GenerateViolations $generateViolations;

    private ExtractPack $extractPack;

    private ModuleRepository $moduleRepository;

    public function __construct(
        XCart $XCart,
        CoreHashRetriever $coreHashRetriever,
        ModuleHashRetriever $moduleHashRetriever,
        ActualFilesRetriever $actualFilesRetriever,
        GenerateViolations $generateViolations,
        ExtractPack  $extractPack,
        ModuleRepository $moduleRepository
    ) {
        $this->XCart                = $XCart;
        $this->coreHashRetriever    = $coreHashRetriever;
        $this->moduleHashRetriever  = $moduleHashRetriever;
        $this->actualFilesRetriever = $actualFilesRetriever;
        $this->generateViolations   = $generateViolations;
        $this->extractPack          = $extractPack;
        $this->moduleRepository     = $moduleRepository;
    }

    /**
     * @throws GetHashException
     * @throws \JsonException
     */
    public function __invoke(array $modulesToUpgrade, $isLoaded = false): array
    {
        if ($isLoaded) {
            return $this->calculateLoadedFilesToMove($modulesToUpgrade);
        }

        $upgradeEntries = $this->calculateModuleUpgradeEntries($modulesToUpgrade);

        if (isset($modulesToUpgrade[Marketplace::CORE_MODULE_ID])) {
            $coreVersion = $modulesToUpgrade[Marketplace::CORE_MODULE_ID]['version'];

            $upgradeEntries[Marketplace::CORE_MODULE_ID] = $this->calculateCoreUpgradeEntries($coreVersion);
        }

        return $upgradeEntries;
    }

    /**
     * @throws GetHashException
     * @throws \JsonException
     */
    private function calculateCoreUpgradeEntries(array $coreVersion, $packagePath = null): array
    {
        $coreKnown = ($this->coreHashRetriever)($coreVersion, $packagePath);

        if (isset($coreKnown['message'])) {
            throw GetHashException::fromGetCoreHashResponseWithError($coreKnown['message']);
        }

        $coreActual = ($this->coreHashRetriever)(
            Marketplace::explodeVersion($this->XCart->getCoreVersion())
        );

        if (isset($coreKnown['message'])) {
            throw GetHashException::fromGetCoreHashResponseWithError($coreActual['message']);
        }

        return ($this->generateViolations)($coreKnown, $coreActual);
    }

    /**
     * @throws GetHashException
     * @throws \JsonException
     */
    private function calculateLoadedFilesToMove(array $modulesToUpgrade): array
    {
        $moduleFilesToMove = [];
        $packsPath = $this->XCart->getPacksPath();

        foreach ($modulesToUpgrade as $module) {
            [$moduleId, $version] = explode(':', $module);
            [$author, $name]      = explode('-', $moduleId);

            $packagePath      = "{$packsPath}{$moduleId}-v{$version}";
            $loadedModulePath = "{$packagePath}/modules/{$author}/{$name}";
            $modulePath       = "{$this->XCart->getSourcePath()}modules/{$author}/{$name}";

            ($this->extractPack)($packagePath);

            if ($moduleId === Marketplace::CORE_MODULE_ID) {
                $moduleFilesToMove[$moduleId] = $this->calculateCoreUpgradeEntries(
                    Marketplace::explodeVersion($version),
                    $packagePath
                );
            } else {
                $loadedModuleHash = ($this->actualFilesRetriever)($loadedModulePath, $moduleId);
                $actual                       = ($this->actualFilesRetriever)($modulePath, $moduleId);
                $moduleFilesToMove[$moduleId] = ($this->generateViolations)($loadedModuleHash, $actual);
            }
        }

        return $moduleFilesToMove;
    }

    /**
     * @throws GetHashException
     */
    private function calculateModuleUpgradeEntries(array $modulesToUpgrade): array
    {
        $moduleUpgradeEntries = [];

        $moduleHash = ($this->moduleHashRetriever)(
            $this->getModuleVersionHashes($modulesToUpgrade)
        );

        foreach ($moduleHash as $moduleId => $data) {
            $known = $data['result'] ?? null;
            if ($known) {
                $actual = $this->getModuleCurrentVersionHashes($moduleId);

                $moduleUpgradeEntries[$moduleId] = ($this->generateViolations)($known, $actual);
            } else {
                throw GetHashException::fromGetAddonHashBatchResponseWithError($moduleId, $data['error']);
            }
        }

        return $moduleUpgradeEntries;
    }

    /**
     * @throws GetHashException
     */
    private function getModuleCurrentVersionHashes(string $moduleId): array
    {
        $actual = [];
        $module = $this->moduleRepository->findByModuleId($moduleId);
        $actualModuleVersionHash = ($this->moduleHashRetriever)(
            [$moduleId => Marketplace::getModuleVersionHash($module->getAuthor(), $module->getName(), $module->getVersion())]
        );

        if (!empty($actualModuleVersionHash)) {
            $actual = $actualModuleVersionHash[$moduleId]['result'] ?? null;
        }

        if ($actual === null) {
            throw GetHashException::fromGetAddonHashBatchResponseWithError($moduleId, $actualModuleVersionHash[$moduleId]['error']);
        }

        return $actual;
    }

    private function getModuleVersionHashes(array $modulesToUpgrade): array
    {
        $moduleVersionHashes = [];
        foreach ($modulesToUpgrade as $moduleId => $upgrade) {
            if ($moduleId === Marketplace::CORE_MODULE_ID) {
                continue;
            }

            $version = $upgrade['version'];

            $moduleVersionHashes[$moduleId] = Marketplace::getModuleVersionHash(
                $upgrade['author'],
                $upgrade['name'],
                Marketplace::implodeVersion($version)
            );
        }

        return $moduleVersionHashes;
    }
}
