<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Upgrade;

use App\Domain\XCart;
use App\Exception\DownloadPackException;
use App\Marketplace\Marketplace;
use App\Operation\Build\ExtractPack;
use App\Operation\Download\DownloaderInterface;
use Symfony\Component\Filesystem\Filesystem;

final class DownloadPacks
{
    private Filesystem $filesystem;

    private DownloaderInterface $downloader;

    private ExtractPack $extractPack;

    private string $packsDir;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem,
        DownloaderInterface $downloader,
        ExtractPack $extractPack
    ) {
        $this->packsDir = $XCart->getPacksPath();

        $this->filesystem  = $filesystem;
        $this->downloader  = $downloader;
        $this->extractPack = $extractPack;
    }

    /**
     * @throws DownloadPackException
     */
    public function __invoke(array $modulesToUpgrade = []): array
    {
        $packPaths = [];
        $this->filesystem->mkdir($this->packsDir);

        foreach ($modulesToUpgrade as $moduleId => $upgrade) {
            $upgradeVersion = Marketplace::implodeVersion($upgrade['version']);
            $packagePath    = "{$this->packsDir}{$moduleId}-v{$upgradeVersion}";

            $this->downloader->download($moduleId, $upgrade, "{$packagePath}.tar.gz");

            ($this->extractPack)($packagePath);

            $packPaths[] = $packagePath;
        }

        return $packPaths;
    }
}
