<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Upgrade;

use App\Domain\XCart;
use App\Exception\CheckUpgradeRequirementsException;
use App\Marketplace\Marketplace;

final class CheckUpgradeRequirements
{
    private XCart $XCart;

    private array $minPHPVersions;

    private $serverPhpversion;

    public function __construct(
        XCart $XCart,
        array $minPHPVersions
    ) {
        $this->XCart = $XCart;

        $serverPhpVersion     = $this->serverPhpversion = phpversion();
        $this->minPHPVersions = array_filter($minPHPVersions,
            static fn($minPHPVersion) => version_compare($serverPhpVersion, $minPHPVersion, '<')
        );

        // must be to work quick-return-from-array in getPHPMinRequiredVersion()
        krsort($this->minPHPVersions);
    }

    /**
     * @throws CheckUpgradeRequirementsException
     */
    public function __invoke(array $modulesToUpgrade): void
    {
        $failedResult = $this->phpVersionCheck($modulesToUpgrade);
        if ($failedResult) {
            throw CheckUpgradeRequirementsException::fromUnsupportedPHPVersion(
                "{$failedResult['coreVersionFrom']}->{$failedResult['coreVersionTo']}", $failedResult['phpMinRequiredVersion']
            );
        }
    }

    /**
     * @throws \Exception
     */
    public function phpVersionCheck(array $modulesToUpgrade): array
    {
        if (
            empty($this->minPHPVersions)
            || empty($modulesToUpgrade)
            || !isset($modulesToUpgrade[Marketplace::CORE_MODULE_ID])
        ) {
            // all $this->minPHPVersions meet the current server version. Nothing 2 check
            return []; // test is OK
        }

        $upgrade = $modulesToUpgrade[Marketplace::CORE_MODULE_ID];

        $coreVersionTo         = Marketplace::implodeVersion($upgrade['version']);
        $phpMinRequiredVersion = $this->getPHPVersionRequiredForThisCore($coreVersionTo, $this->minPHPVersions);

        if (
            empty($phpMinRequiredVersion)
            || version_compare($this->serverPhpversion, $phpMinRequiredVersion, '>=')
        ) {
            return []; // test is OK
        }

        $coreVersionFrom = $this->XCart->getCoreVersion();

        // test failed. Return related data
        return [
            'coreVersionFrom'       => $coreVersionFrom,
            'coreVersionTo'         => $coreVersionTo,
            'phpMinRequiredVersion' => preg_replace('/^(\d+\.\d+)\.0+$/', "\\1", $phpMinRequiredVersion),
        ];
    }

    private function getPHPVersionRequiredForThisCore(string $actualVersionTo, array $minPHPVersions): string
    {
        foreach ($minPHPVersions as $lowerBoundVersionTo => $minPHPVersion) {
            if (version_compare($actualVersionTo, $lowerBoundVersionTo, '>=')) {
                return $minPHPVersion;
            }
        }

        return '';
    }
}
