<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Upgrade;

use App\Domain\XCart;
use App\Exception\CheckPermissionsException;

final class CheckPermissions
{
    private string $sourcePath;

    public function __construct(XCart $XCart)
    {
        $this->sourcePath = $XCart->getSourcePath();
    }

    /**
     * @throws CheckPermissionsException
     */
    public function __invoke(array $upgradeEntries): void
    {
        foreach ($upgradeEntries as $moduleId => $moduleUpgradeEntries) {
            foreach ($moduleUpgradeEntries as $upgradeEntry) {
                $filePath = "{$this->sourcePath}{$upgradeEntry['filepath']}";

                if (in_array($upgradeEntry['type'], ['modified', 'added'], true)) {
                    if (!is_writable($filePath)) {
                        throw CheckPermissionsException::fromNonWritableFile($filePath, $moduleId);
                    }
                } elseif ($upgradeEntry['type'] === 'removed') {
                    $dir = $this->getNearestExistingDirectory($filePath);

                    if (!is_writable($dir)) {
                        throw CheckPermissionsException::fromNonWritableDir($dir, $moduleId);
                    }
                }
            }
        }
    }

    private function getNearestExistingDirectory(string $path): string
    {
        do {
            $path = dirname($path);
        } while (
            !is_dir($path)
            && ($path !== $this->sourcePath)
        );

        return $path;
    }
}
