<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build;

use App\Exception\AdminCredentialsException;

final class ParseCredentials
{
    protected function isAllowedAdminProfileLanguage(string $lngCode): bool
    {
        return in_array($lngCode, ['en', 'es'], true);
    }

    /**
     * @throws AdminCredentialsException
     */
    public function __invoke(string $adminCredentials, string $lang = 'en'): array
    {
        $this->checkAdminCredentialsFormat($adminCredentials);

        [$login, $password] = explode(':', $adminCredentials, 2);

        $this->checkLoginFormat($login);
        $this->checkPasswordFormat($password);

        $encryptedPassword = hash('md5', $password);

        return [
            'login'    => $login,
            'password' => $encryptedPassword,
            'lang'     => $this->isAllowedAdminProfileLanguage($lang) ? $lang : 'en'
        ];
    }

    /**
     * @throws AdminCredentialsException
     */
    private function checkAdminCredentialsFormat(string $adminCredentials): void
    {
        if (strpos($adminCredentials, ':') === false) {
            throw AdminCredentialsException::fromWrongCredentialsFormat();
        }
    }

    /**
     * @throws AdminCredentialsException
     */
    private function checkLoginFormat(string $login): void
    {
        $isCorrectLogin = preg_match(
            '/^[a-z0-9!#$%&\'*+\/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&\'*+\/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z](?:[a-z0-9-]*[a-z0-9])?\s*$/i',
            $login
        );

        if (!$isCorrectLogin) {
            throw AdminCredentialsException::fromWrongLoginFormat();
        }
    }

    /**
     * @throws AdminCredentialsException
     */
    private function checkPasswordFormat(string $password): void
    {
        $passwordLength = strlen($password);

        if ($passwordLength < 8 || $passwordLength > 64) {
            throw AdminCredentialsException::fromWrongPasswordFormat();
        }
    }
}
