<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\InstallModules;

use App\Domain\XCart;
use App\Exception\DownloadModulesException;
use App\Exception\DownloadPackException;
use App\Marketplace\Marketplace;
use App\Operation\Build\ExtractPack;
use App\Operation\Download\DownloaderInterface;
use App\Repository\ModuleRepository;
use Symfony\Component\Filesystem\Filesystem;

final class DownloadPacks
{
    private string $packsDir;

    private Filesystem $filesystem;

    private DownloaderInterface $downloader;

    private Marketplace $marketplace;

    private ModuleRepository $moduleRepository;

    private ExtractPack $extractPack;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem,
        DownloaderInterface $downloader,
        Marketplace $marketplace,
        ModuleRepository $moduleRepository,
        ExtractPack $extractPack
    ) {
        $this->packsDir = $XCart->getPacksPath();

        $this->filesystem       = $filesystem;
        $this->downloader       = $downloader;
        $this->marketplace      = $marketplace;
        $this->moduleRepository = $moduleRepository;
        $this->extractPack      = $extractPack;
    }

    /**
     * @throws DownloadPackException
     * @throws DownloadModulesException
     */
    public function __invoke(array $modulesToInstall = [], array $params = []): array
    {
        $packPaths = [];
        $this->filesystem->mkdir($this->packsDir);

        foreach ($modulesToInstall as $moduleId) {
            if ($this->moduleRepository->findByModuleId($moduleId)) {
                throw DownloadModulesException::fromAlreadyInstalledModule($moduleId);
            }

            [$author, $name] = explode('-', $moduleId);
            $module = $this->marketplace->getModule($author, $name);

            if (!$module) {
                throw DownloadModulesException::fromUnknownModule($moduleId);
            }

            $moduleVersion = Marketplace::implodeVersion($module['version']);
            $packagePath   = "{$this->packsDir}{$moduleId}-v{$moduleVersion}";
            $upgradeSpec   = array_merge($module, $params);

            // $module['moduleId'] contains md5 hash for the string "{$author}.{$name}.{$version}"
            $this->downloader->download($moduleId, $upgradeSpec, "{$packagePath}.tar.gz");

            ($this->extractPack)($packagePath);

            $packPaths[$moduleId] = $packagePath;
        }

        return $packPaths;
    }
}
