<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\InstallModules;

use App\Domain\XCart;
use App\Exception\CheckPermissionsException;
use Symfony\Component\Filesystem\Filesystem;

final class CheckPermissions
{
    private string $modulesPath;

    private Filesystem $filesystem;

    public function __construct(
        XCart      $XCart,
        Filesystem $filesystem
    ) {
        $this->modulesPath = $XCart->getModulesPath();
        $this->filesystem  = $filesystem;
    }

    /**
     * @throws CheckPermissionsException
     */
    public function __invoke(array $modulesToInstall): void
    {
        foreach ($modulesToInstall as $moduleId) {
            [$author, $name] = explode('-', $moduleId);

            $moduleDir = "{$this->modulesPath}{$author}/{$name}";
            if ($this->filesystem->exists($moduleDir)) {
                throw CheckPermissionsException::fromAlreadyExistedDir($moduleDir, $moduleId);
            }

            $authorDir = "{$this->modulesPath}{$author}";
            if ($this->filesystem->exists($authorDir)) {
                if (!is_writable($authorDir)) {
                    throw CheckPermissionsException::fromNonWritableDir($authorDir, $moduleId);
                }
            } elseif (!is_writable($this->modulesPath)) {
                throw CheckPermissionsException::fromNonWritableDir($this->modulesPath, $moduleId);
            }
        }
    }
}
