<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build;

use App\Domain\ModuleDomain;
use App\Entity\Module;
use App\Exception\BuildException;
use App\Exception\CircularDependencyException;
use App\Logic\CheckRequirements;
use App\Repository\ModuleRepository;
use JsonException;
use MJS\TopSort\CircularDependencyException as TopSortCircularDependencyException;
use MJS\TopSort\ElementNotFoundException as TopSortElementNotFoundException;
use RuntimeException;

final class GenerateTransitions
{
    private CheckRequirements $checkRequirements;

    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    private array $modules = [];

    public function __construct(
        CheckRequirements $checkRequirements,
        ModuleDomain $moduleDomain,
        ModuleRepository  $moduleRepository
    ) {
        $this->checkRequirements = $checkRequirements;
        $this->moduleDomain      = $moduleDomain;
        $this->moduleRepository  = $moduleRepository;
    }

    /**
     * @throws BuildException
     * @throws JsonException
     * @throws RuntimeException
     * @throws TopSortCircularDependencyException
     * @throws TopSortElementNotFoundException
     * @throws CircularDependencyException
     */
    public function __invoke(
        array $modulesToEnable = [],
        array $modulesToDisable = [],
        array $modulesToRemove = [],
        array $modulesToInstall = []
    ): array {
        $modulesToEnable      = $this->getModulesData($modulesToEnable);
        $modulesToDisable     = $this->getModulesData($modulesToDisable);
        $modulesToRemove      = $this->getModulesData($modulesToRemove);
        $modulesToInstallObjs = $this->getModulesData($modulesToInstall);

        $transitions           = [];
        $modulesForTransitions = $this->getModulesForTransitions($modulesToEnable, $modulesToDisable, $modulesToRemove, $modulesToInstallObjs);

        foreach ($modulesForTransitions['toEnable'] as $moduleId) {
            if (in_array($moduleId, $modulesToInstall, true)) {
                continue;
            }
            $module = $this->getModule($moduleId);

            if ($module && !$module->isEnabled()) {
                $transitions[$module->getModuleId()] = [
                    'stateBefore'      => $module->getState(),
                    'stateAfter'       => Module::STATE_ENABLED,
                    'transitionType'   => 'enable',
                    'moduleName'       => $module->getMetaData()['moduleName'],
                    'showSettingsForm' => $module->getMetaData()['showSettingsForm'],
                ];

                // delayed upgrade (module was upgraded in disabled state and then this module is enabled)
                if ($module->getState() === Module::STATE_INSTALLED) {
                    $transitions[$module->getModuleId()]['versionFrom'] = $module->getLastEnabledVersion();
                    $transitions[$module->getModuleId()]['versionTo']   = $module->getVersion();
                }
            }
        }

        foreach ($modulesForTransitions['toDisable'] as $moduleId) {
            $module = $this->getModule($moduleId);

            if ($module && $module->isEnabled()) {
                $transitions[$module->getModuleId()] = [
                    'stateBefore'    => $module->getState(),
                    'stateAfter'     => Module::STATE_INSTALLED,
                    'transitionType' => 'disable',
                    'moduleName'     => $module->getMetaData()['moduleName'],
                ];
            }
        }

        foreach ($modulesForTransitions['toRemove'] as $moduleId) {
            $module = $this->moduleRepository->findByModuleId($moduleId);
            if ($module) {
                $transitions[$module->getModuleId()] = [
                    'stateBefore'    => $module->getState(),
                    'stateAfter'     => Module::STATE_REMOVED,
                    'transitionType' => 'remove',
                    'moduleName'     => $module->getMetaData()['moduleName'],
                ];
            }
        }

        foreach ($modulesToInstall as $moduleId) {
            $module = $this->moduleRepository->findByModuleId($moduleId);
            $transitions[$moduleId] = [
                'stateBefore'    => Module::STATE_NOT_INSTALLED,
                'stateAfter'     => Module::STATE_ENABLED,
                'transitionType' => 'install',
                'showSettingsForm' => $module ? $module->getMetaData()['showSettingsForm'] : false,
            ];
        }

        return $transitions;
    }

    /**
     * @returns Module[]
     * @throws RuntimeException
     */
    private function getModulesData(array $modules): array
    {
        $result = [];
        foreach ($modules as $key => $moduleId) {
            $module = $this->getModule($moduleId);

            $this->checkModuleExists($module, $moduleId);

            $result[$key] = $module;
        }

        return $result;
    }

    /**
     * @param Module[] $modulesToEnable
     * @param Module[] $modulesToDisable
     * @param Module[] $modulesToRemove
     * @param Module[] $modulesToInstall
     *
     * @return array
     *
     * @throws BuildException
     * @throws CircularDependencyException
     * @throws JsonException
     * @throws TopSortCircularDependencyException
     * @throws TopSortElementNotFoundException
     */
    private function getModulesForTransitions(array $modulesToEnable, array $modulesToDisable, array $modulesToRemove, array $modulesToInstall): array
    {
        $this->setDependencyMap();

        $this->checkRequirements->setEnabledModuleIds(
            $this->moduleRepository->getEnabledIds()
        );

        $this->checkRequirements->setSortedEnabledSkinModuleIds(
            $this->moduleRepository->getSortedEnabledSkinModuleIds()
        );

        $this->addRequirementsAction($modulesToDisable, 'disable');
        $this->addRequirementsAction($modulesToEnable, 'enable');
        $this->addRequirementsAction($modulesToRemove, 'remove');
        $this->addRequirementsAction($modulesToInstall, 'install');

        $requirements = $this->checkRequirements->checkActions();

        if ($requirements->getFailedActions()) {
            throw BuildException::fromNotAllowedModuleAction($requirements->getFailedActions(), $requirements->getCantReasons());
        }

        return [
            'toEnable'  => $requirements->getToEnable(),
            'toDisable' => $requirements->getToDisable(),
            'toRemove'  => $requirements->getToRemove(),
        ];
    }

    /**
     * @param Module[]  $modules
     *
     * @throws CircularDependencyException
     */
    private function addRequirementsAction(array $modules, string $actionType): void
    {
        foreach ($modules as $module) {
            [$majorVer1, $majorVer2, $minorVersion, $buildVersion] = explode('.', $module->getVersion());
            $this->checkRequirements->addAction(
                $actionType,
                $module->getModuleId(),
                $module->isSkin(),
                $module->getMetaData()['canDisable'],
                $module->getMetaData()['minorRequiredCoreVersion']
                    ? "{$majorVer1}.{$majorVer2}.{$module->getMetaData()['minorRequiredCoreVersion']}.0"
                    : '',
                $module->getMetaData()['dependsOn'] ?? []
            );

            // The module may be missed in the DataBase, so it is required to add a dependencies to the map
            $this->checkRequirements->setDependencyMap(
                [
                    $module->getModuleId() => [
                        $module->getMetaData()['dependsOn'] ?? [],
                        $module->getMetaData()['incompatibleWith'] ?? [],
                    ]
                ]
            );
        }
    }

    /**
     * @throws BuildException
     */
    private function setDependencyMap(): void
    {
        $dependencyMap = [];

        foreach ($this->moduleRepository->getAllModules() as $module) {
            /** @var Module $module */
            $metadata = $module->getMetaData();

            $dependencyMap[$module->getModuleId()] = [
                $metadata['dependsOn'],
                $metadata['incompatibleWith'],
            ];
        }

        try {
            $this->checkRequirements->setDependencyMap($dependencyMap);
        } catch (CircularDependencyException $e) {
            throw BuildException::fromCyclesInDependencies($e);
        }
    }

    /**
     * @throws RuntimeException
     */
    private function checkModuleExists(?Module $module, string $moduleId): void
    {
        if ($module) {
            // for cases like that:
            // - $module->getModuleId() returns CDev-EGoods
            // - $moduleId returns CDev-Egoods)
            if ($module->getModuleId() !== $moduleId) {
                throw new RuntimeException("Addon {$moduleId} is not found. Did you mean {$module->getModuleId()}?");
            }
        } else {
            throw new RuntimeException("Addon {$moduleId} is not found.");
        }
    }

    private function getModule(string $moduleId): ?Module
    {
        if (!isset($this->modules[$moduleId])) {
            $module = $this->moduleRepository->findByModuleId($moduleId);

            // In case the module source code was copied manually we need to read its info from the file
            try {
                if (
                    !$module
                    && ($moduleInfo = $this->moduleDomain->readModuleInfo($moduleId))
                ) {
                    $module = $this->moduleRepository->createModuleFromArray($moduleInfo);
                }
            } catch (\Exception $e) {
            }

            if ($module) {
                $this->modules[$moduleId] = $module;
            }
        }

        return $this->modules[$moduleId] ?? null;
    }
}
