<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Configuration;

use App\Repository\ModuleRepository;
use JsonException;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;

final class GenerateSkinModel
{
    private const PATH_PATTERN = '{{type}}';

    private array $layoutModel;

    private ModuleRepository $moduleRepository;

    private array $layoutModelPaths = [];

    public function __construct(
        array $layoutModel,
        ModuleRepository $moduleRepository
    ) {
        $this->layoutModel      = $layoutModel;
        $this->moduleRepository = $moduleRepository;
    }

    public function __invoke(): array
    {
        $skinModel = [];
        foreach ($this->layoutModel as $interface => $config) {
            foreach ($config as $zone => $spec) {
                $skinModel[$interface][$zone] = $this->generateSkinPaths($interface, $zone);
            }
        }

        return ['parameters' => ['xcart.skin_model' => $skinModel]];
    }

    private function generateSkinPaths(string $interface, string $zone)
    {
        $modelPaths = $this->getLayoutModelPaths($interface, $zone);
        $result     = [];

        foreach ($modelPaths as $modelPath) {
            $preparedPaths = array_map(static function ($modulePath) use ($modelPath) {
                return $modulePath . '/' . static::PATH_PATTERN . '/' . $modelPath;
            }, $this->getModulePaths());

            $preparedPaths[] = static::PATH_PATTERN . '/' . $modelPath;

            array_unshift($preparedPaths, $this->getCustomSkinPaths($modelPath));

            $result[] = $preparedPaths;
        }

        return array_merge(...array_values($result));
    }

    /**
     * @throws CircularDependencyException
     * @throws ElementNotFoundException
     * @throws JsonException
     */
    private function getModulePaths()
    {
        $moduleIds = array_unique(
            array_merge(
                $this->moduleRepository->getSortedEnabledSkinModuleIds(),
                $this->moduleRepository->getEnabledIds()
            )
        );

        return array_values(
            array_map(static function ($moduleId) {
                return 'modules/' . str_replace('-', '/', $moduleId);
            }, $moduleIds)
        );
    }

    private function getLayoutModelPaths(string $interface, string $zone)
    {
        if (isset($this->layoutModelPaths[$interface][$zone])) {
            return $this->layoutModelPaths[$interface][$zone];
        }

        $spec = $this->layoutModel[$interface][$zone];

        $result = [
            $spec['path']
        ];

        foreach ($spec['fallback'] ?? [] as $fallback) {
            [$fallbackInterface, $fallbackZone] = explode('.', $fallback);

            foreach ($this->getLayoutModelPaths($fallbackInterface, $fallbackZone) as $path) {
                $result[] = $path;
            }
        }

        return $this->layoutModelPaths[$interface][$zone] = $result;
    }

    private function getCustomSkinPaths(string $modelPath): string
    {
        $parts = explode('/', $modelPath);
        $interface = $parts[0] ?? '';
        $zone = end($parts);

        return self::PATH_PATTERN . "/custom_{$interface}/" . $zone;
    }
}
