<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build\Configuration;

use App\Domain\XCart;
use App\Repository\ModuleRepository;
use Symfony\Component\Filesystem\Path;

final class GenerateAutoloader
{
    private XCart $XCart;

    private ModuleRepository $moduleRepository;

    private string $kernelEnvironment;

    public function __construct(
        XCart $XCart,
        ModuleRepository $moduleRepository,
        string $kernelEnvironment
    ) {
        $this->XCart             = $XCart;
        $this->moduleRepository  = $moduleRepository;
        $this->kernelEnvironment = $kernelEnvironment;
    }

    public function __invoke(): string
    {
        $sourcePath = $this->XCart->getSourcePath();
        $targetPath = "realpath(__DIR__ . '/../" . Path::makeRelative($this->XCart->getCodeCachePath(), $sourcePath) . "') . '/'";

        $enabledModules = $this->moduleRepository->getSortedEnabledIds();

        $moduleAutoloaders = "[\n";
        foreach ($this->getModuleAutoloaders($sourcePath, $enabledModules) as $autoloader) {
            $moduleAutoloaders .= "    realpath(__DIR__ . '/../$autoloader'),\n";
        }
        $moduleAutoloaders .= ']';

        $sources = "[\n";
        foreach ($this->getSources($enabledModules) as $namespace => $path) {
            $sources .= "        '$namespace' => realpath(__DIR__ . '/../$path'),\n";
        }
        $sources .= "    ]";

        $modulesStr = "[\n";
        foreach ($enabledModules as $module) {
            $modulesStr .= "        '$module',\n";
        }
        $modulesStr .= "    ]";

        $dependencies = "[\n";
        foreach ($this->getDependencies() as $moduleId => $moduleDependencies) {
            $dependencies .= "        '$moduleId' => ['" . implode("', '", $moduleDependencies) . "'],\n";
        }
        $dependencies .= "    ]";

        if ($this->kernelEnvironment === 'dev') {
            $mainAutoloader = <<<"PHP"

class_alias('\\XCart\\Framework\\ApiPlatform\\Core\\Util\\ReflectionClassRecursiveIterator', '\\ApiPlatform\\Util\\ReflectionClassRecursiveIterator');
class_alias('\\XCart\\Framework\\ApiPlatform\\Core\\Util\\ReflectionClassRecursiveIterator', '\\ApiPlatform\\Core\\Util\\ReflectionClassRecursiveIterator');
class_alias('\\XCart\\Doctrine\\Common\\Annotations\\PhpParser', '\\Doctrine\\Common\\Annotations\\PhpParser');
class_alias('\\XCart\\Doctrine\\Persistence\\Mapping\\Driver\\ColocatedMappingDriver', '\\Doctrine\\Persistence\\Mapping\\Driver\\ColocatedMappingDriver');
class_alias('\\XCart\\Symfony\\Component\\DependencyInjection\\Dumper\\PhpDumper', '\\Symfony\\Component\\DependencyInjection\\Dumper\\PhpDumper');

\\XCart\\Framework\\AutoloaderManager::registerDevelopment();
PHP;

        } else {
            $mainAutoloader = <<<"PHP"
\\XCart\\Framework\\AutoloaderManager::registerProduction();
PHP;
        }

        return <<<"PHP"
<?php

\\XCart\\Framework\\AutoloaderManager::setDevelopmentAutoloader((static function (): \\XCart\\Extender\\Autoloader\\AutoloaderInterface {
    \$xcartExtender = new \\XCart\\Extender\\Extender();

    \$xcartExtender
    ->setSources({$sources})
    ->setTargetRoot({$targetPath})
    ->setModules({$modulesStr})
    ->setDependencies({$dependencies})
    ->addSubscribers();

    \$xcartExtender->setDeveloperMode();

    return new \\XCart\\Extender\\Autoloader\\DevelopmentAutoloader(
        \$xcartExtender->getBuilder(),
        \$xcartExtender->getSourceMap(),
        new \\XCart\\Extender\\Domain\\TargetMap({$targetPath}),
    );
})());

\\XCart\\Framework\\AutoloaderManager::setProductionAutoloader((static function (): \\XCart\\Extender\\Autoloader\\AutoloaderInterface {
    \$xcartExtender = new \\XCart\\Extender\\Extender();

    \$xcartExtender->setSources({$sources});

    return new \\XCart\\Extender\\Autoloader\\DynamicAutoloader(
        \$xcartExtender->getSourceMap(),
        new \\XCart\\Extender\\Domain\\TargetMap({$targetPath}),
    );
})());

\\XCart\\Framework\\AutoloaderManager::setModuleAutoloaders({$moduleAutoloaders});

{$mainAutoloader}

\\XCart\\Framework\\AutoloaderManager::registerModuleAutoloaders();

PHP;
    }

    private function getSources(array $modules): array
    {
        $result = [];

        $modulesSources = $this->moduleRepository->getSources();
        foreach ($modulesSources as $moduleId => [$namespace, $source]) {
            if (
                in_array($moduleId, $modules, true)
                && is_dir($this->XCart->getSourcePath() . 'modules/' . $source . '/src')
            ) {
                $result[$namespace] = 'modules/' . $source . '/src';
            }
        }

        $result['XLite'] =  'classes/XLite';

        return $result;
    }

    private function getDependencies(): array
    {
        $result = [];

        $dependencyMap = $this->moduleRepository->getDependencyMap();
        foreach ($dependencyMap as $id => $dependency) {
            if (!empty($dependency[0])) {
                $result[$id] = $dependency[0];
            }
        }

        return $result;
    }

    /**
     * @throws \JsonException
     */
    private function getModuleAutoloaders(string $path, array $modules): array
    {
        $result = [];
        foreach ($this->moduleRepository->getAutoloaders($modules) as $autoloaders) {
            foreach (json_decode($autoloaders['autoloader'], true, 512, JSON_THROW_ON_ERROR) ?? [] as $autoloader) {
                $result[] = Path::makeRelative($path . 'modules/' . $autoloaders['path'] . '/' . $autoloader, $this->XCart->getSourcePath());
            }
        }

        return $result;
    }
}
