<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build;

use App\Domain\ModuleDomain;
use App\Exception\CheckPermissionsException;
use App\Exception\RemoveModulesException;
use App\Repository\ModuleRepository;
use Symfony\Component\Filesystem\Filesystem;

final class CheckRemovePermissions
{
    private Filesystem $filesystem;

    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    public function __construct(
        Filesystem $filesystem,
        ModuleDomain $moduleDomain,
        ModuleRepository $moduleRepository
    ) {
        $this->filesystem       = $filesystem;
        $this->moduleDomain     = $moduleDomain;
        $this->moduleRepository = $moduleRepository;
    }

    /**
     * @throws CheckPermissionsException
     * @throws RemoveModulesException
     */
    public function __invoke(array $modulesToRemove = []): void
    {
        foreach ($modulesToRemove as $moduleId) {
            $path = $this->moduleDomain->getModulesCodePath() . str_replace('-', '/', $moduleId);
            if (!is_writable($path)) {
                throw CheckPermissionsException::fromNonWritableDir($path, $moduleId);
            }

            if (!$this->moduleRepository->findByModuleId($moduleId)) {
                throw RemoveModulesException::fromAlreadyRemovedModule($moduleId);
            }
        }
    }
}
