<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Operation\Build;

use App\Domain\XCart;
use App\Repository\ModuleRepository;
use XCart\Extender\Extender;

final class BuildCodeCache
{
    private XCart $XCart;

    private ModuleRepository $moduleRepository;

    private Extender $extender;

    public function __construct(
        XCart $XCart,
        ModuleRepository $moduleRepository,
        Extender $extender
    ) {
        $this->XCart            = $XCart;
        $this->moduleRepository = $moduleRepository;
        $this->extender         = $extender;
    }

    public function __invoke()
    {
        $enabledModules = $this->moduleRepository->getEnabledIds();
        $sources        = $this->getSources($this->XCart->getSourcePath(), $enabledModules);
        $dependencies   = $this->getDependencies();

        $this->extender
            ->setSources($sources)
            ->setTargetRoot($this->XCart->getCodeCachePath())
            ->setModules($enabledModules)
            ->setDependencies($dependencies)
            ->addSubscribers();

        $sourceMap = $this->extender->getSourceMap();
        $builder   = $this->extender->getBuilder();

        foreach ($sourceMap->getFiles() as $file) {
            $builder->buildByPath($file);
        }
    }

    private function getSources(string $path, array $modules): array
    {
        $result = [];

        $modulesSources = $this->moduleRepository->getSources();
        foreach ($modulesSources as $moduleId => [$namespace, $source]) {
            if (
                in_array($moduleId, $modules, true)
                && is_dir($path . 'modules/' . $source . '/src')
            ) {
                $result[$namespace] = $path . 'modules/' . $source . '/src';
            }
        }

        $result['XLite'] = $path . 'classes/XLite';

        return $result;
    }

    private function getDependencies(): array
    {
        $result = [];

        $dependencyMap = $this->moduleRepository->getDependencyMap();
        foreach ($dependencyMap as $id => $dependency) {
            if (!empty($dependency[0])) {
                $result[$id] = $dependency[0];
            }
        }

        return $result;
    }
}
