<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Marketplace;

use App\Domain\ShopURLs;
use App\Domain\XCart;
use App\Repository\DataRepository;
use App\Repository\LicenseRepository;
use Psr\Log\LoggerInterface;
use stdClass;
use Symfony\Contracts\Cache\CacheInterface;
use XCartMarketplace\Connector\Client;
use XCartMarketplace\Connector\Config;
use XCartMarketplace\Connector\Exceptions\ClientException;
use XCartMarketplace\Connector\Exceptions\RequestValidationException;

final class MarketplaceClient
{
    private ShopURLs $shopURLs;

    private XCart $XCart;

    private DataRepository $dataRepository;

    private LicenseRepository $licenseRepository;

    private LoggerInterface $logger;

    private ?Client $client = null;

    private array $config;

    private int $timeout;

    private int $connectTimeout;

    private CacheInterface $cache;

    public function __construct(
        $config,
        $timeout,
        $connectTimeout,
        XCart $XCart,
        ShopURLs $shopURLs,
        DataRepository $dataRepository,
        LicenseRepository $licenseRepository,
        CacheInterface $cache,
        LoggerInterface $logger
    ) {
        $this->XCart             = $XCart;
        $this->dataRepository    = $dataRepository;
        $this->licenseRepository = $licenseRepository;
        $this->logger            = $logger;
        $this->cache             = $cache;
        $this->shopURLs          = $shopURLs;
        $this->config            = $config;
        $this->timeout           = (int) $timeout;
        $this->connectTimeout    = (int) $connectTimeout;
    }

    /**
     * @param string $target         Request target
     * @param array  $params         Request parameters
     * @param bool   $ignoreCache    Flag: true - ignore cache and force sending request
     * @param array  $serviceOptions Additional (service) options, e.g. 'range' for get_addon_pack/get_core_pack
     *
     * @return array|null
     */
    public function retrieve(string $target, array $params = [], bool $ignoreCache = false, array $serviceOptions = []): ?array
    {
        try {
            $this->getClient()->addRequest($target, $params, $ignoreCache, $serviceOptions, $this->timeout, $this->connectTimeout);

            return $this->getClient()->getData()[$target] ?? null;

        } catch (RequestValidationException|ClientException $e) {
            $this->logger->error($e->getMessage());
        }

        return null;
    }

    private function getClient(): Client
    {
        if (!$this->client) {
            $marketplaceConfig = new Config($this->prepareMarketplaceClientConfig((object) $this->config));
            $this->client      = new Client($marketplaceConfig, $this->cache, $this->logger);
        }

        return $this->client;
    }

    private function prepareMarketplaceClientConfig(stdClass $config): array
    {
        $data        = $this->dataRepository->getData();
        $coreLicense = $this->licenseRepository->findCoreLicense();

        return [
            'url'                => $config->url,
            'shopID'             => $this->getShopId($config),
            'shopDomain'         => $this->shopURLs->getShopDomain(),
            'shopURL'            => $this->shopURLs->getShopURL(),
            'currentCoreVersion' => Marketplace::explodeVersion($this->XCart->getCoreVersion()),
            'email'              => $data['xcart.marketplace.admin-email'] ?? '',
            'installation_lng'   => $config->installationLng,
            'shopCountryCode'    => $data['xcart.marketplace.shop-country'] ?? '',
            'affiliateId'        => $config->affiliateId,
            'trial'              => $config->isTrial,
            'cloud'              => false,
            'xcn_license_key'    => $coreLicense ? $coreLicense->getKeyValue() : '',
        ];
    }

    private function getShopId(stdClass $config): string
    {
        return $config->authCode
            ? md5("{$config->authCode}{$config->sharedSecretKey}")
            : '';
    }
}
