<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Marketplace;

use App\Domain\ShopURLs;
use App\Entity\License;
use App\Entity\Module;
use App\Exception\CreateStoreIdentityException;
use App\Exception\GetTokenDataException;
use App\Exception\GetUpgradesException;
use App\Exception\GetVersionInfoException;
use App\Repository\LicenseRepository;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

final class Marketplace
{
    public const CORE_MODULE_ID = 'CDev-Core';

    public const UPGRADE_TYPE_BUILD = 'build';
    public const UPGRADE_TYPE_MINOR = 'minor';
    public const UPGRADE_TYPE_MAJOR = 'major';

    public const UPGRADE_TYPES = [
        self::UPGRADE_TYPE_BUILD,
        self::UPGRADE_TYPE_MINOR,
        self::UPGRADE_TYPE_MAJOR,
    ];

    private bool $isDemoMode;

    private MarketplaceClient $client;

    private MarketplaceStorage $storage;

    private ModuleRepository $moduleRepository;

    private LicenseRepository $licenseRepository;

    private EntityManagerInterface $entityManager;

    private ShopURLs $shopURLs;

    private array $modules = [];

    public function __construct(
        MarketplaceClient $client,
        MarketplaceStorage $storage,
        ModuleRepository $moduleRepository,
        LicenseRepository $licenseRepository,
        EntityManagerInterface $entityManager,
        ShopURLs $shopURLs,
        bool $isDemoMode = false
    ) {
        $this->isDemoMode        = $isDemoMode;
        $this->client            = $client;
        $this->storage           = $storage;
        $this->moduleRepository  = $moduleRepository;
        $this->licenseRepository = $licenseRepository;
        $this->entityManager     = $entityManager;
        $this->shopURLs          = $shopURLs;
    }

    /**
     * @throws Exception
     */
    public function getInstalledModules(): array
    {
        return array_values(array_map(function ($module) {
            /** @var Module $module */
            $version = $this->explodeVersion($module->getVersion());

            return [
                'author'  => $module->getAuthor(),
                'name'    => $module->getName(),
                'major'   => $version['major'],
                'minor'   => $version['minor'],
                'build'   => $version['build'],
                'enabled' => $module->isEnabled(),
            ];
        }, $this->moduleRepository->getAllModules()));
    }

    public function getAllModules(): array
    {
        $params = [
            'modules' => $this->getInstalledModules(),
        ];

        $modules = $this->client->retrieve(
            'get_addons',
            $params
        );

        return $modules['modules'] ?? [];
    }

    public function getModule(string $author, string $name): array
    {
        if (!$this->modules) {
            $this->modules = $this->getAllModules();
        }

        foreach ($this->modules as $module) {
            if (
                $author === $module['author']
                && $name === $module['name']
            ) {
                return $module;
            }
        }

        return [];
    }

    /**
     * @throws GetUpgradesException
     */
    public function getUpgrades(bool $withAdditionalInfo = false): array
    {
        $params = [
            'modules'            => $this->getInstalledModules(),
            'withAdditionalInfo' => $withAdditionalInfo,
        ];

        $response = $this->client->retrieve(
            'get_addons_upgrade',
            $params
        );

        if (isset($response['message'])) {
            throw GetUpgradesException::fromResponseWithError($response['message']);
        }

        return $this->getTransformedUpgradeResponse($response);
    }

    /**
     * @throws GetVersionInfoException
     */
    public function getVersionInfo(array $entities): array
    {
        $params = ['entities' => $entities];

        $response = $this->client->retrieve(
            'get_version_info',
            $params
        );

        if (isset($response['message'])) {
            throw GetVersionInfoException::fromResponseWithError($response['message']);
        }

        return $response;
    }

    public function isStoreIdentityUpdateNeeded(array $params): bool
    {
        $paramsHash              = md5(serialize($params));
        $isUpdateNeeded          = false;
        $storeIdentityParamsHash = $this->storage->getValue('storeIdentityParamsHash');

        if ($storeIdentityParamsHash !== $paramsHash) {
            $isUpdateNeeded = true;
            $this->storage->setValue('storeIdentityParamsHash', $paramsHash);
        }

        return $isUpdateNeeded;
    }

    /**
     * @throws CreateStoreIdentityException
     */
    public function retrieveStoreIdentity(): string
    {
        if ($this->isDemoMode) {
            // don't create store_identity on my.x-cart.com for demostore.x-cart.com
            return '';
        }
        $params = [
            'modules'        => $this->getInstalledModules(),
            'public_key'     => $this->getSecretPublicKey(),
            'addonsLicenses' => $this->licenseRepository->getAddonLicenseKeyValues(),
            'dependencyMap'  => $this->moduleRepository->getDependencyMap(),
            'adminURL'       => $this->shopURLs->getShopURL() . '/service.php/market_module_installer',
            'addonsTokens'   => [],
            'wave'           => $this->getCoreLicense()->getKeyData()['wave'] ?? null,
        ];

        $response = $this->client->retrieve(
            'create_store_identity',
            $params,
            $this->isStoreIdentityUpdateNeeded($params)
        ) ?: [];

        if (isset($response['message'])) {
            throw CreateStoreIdentityException::fromResponseWithError($response['message']);
        }

        return array_pop($response) ?? '';
    }

    public function getSecretPublicKey(): string
    {
        if ($keyPair = $this->storage->getValue('secret_public_keypair')) {
            $keyPair = sodium_base642bin($keyPair, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING);
        } else {
            $keyPair = sodium_crypto_box_keypair();
            $this->storage->setValue('secret_public_keypair', sodium_bin2base64($keyPair, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING));
        }

        $publicKey = sodium_crypto_box_publickey($keyPair);

        return sodium_bin2base64($publicKey, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING);
    }

    private function getTransformedUpgradeResponse($response): array
    {
        // move CORE upgrades to array with BUILD/MINOR/MAJOR key on the first place and unset CORE key
        foreach ($response['core'] ?? [] as $upgradeType => $upgrade) {
            $response[$upgradeType] = [self::CORE_MODULE_ID => $upgrade] + ($response[$upgradeType] ?? []);
        }
        unset($response['core']);

        foreach ($response as $upgradeType => $upgrades) {
            $response[$upgradeType] = array_map(
                static fn($upgrade) => array_merge($upgrade, ['type' => $upgradeType]),
                $upgrades
            );
        }

        // MINOR upgrades must also have BUILD upgrades for the modules that don't have MINOR upgrade
        if (isset($response[self::UPGRADE_TYPE_MINOR])) {
            $response[self::UPGRADE_TYPE_MINOR] = array_merge(
                $response[self::UPGRADE_TYPE_BUILD] ?? [],
                $response[self::UPGRADE_TYPE_MINOR]
            );
        }

        // unset MAJOR upgrades if there are MINOR or BUILD upgrades
        if (
            isset($response[self::UPGRADE_TYPE_BUILD])
            || isset($response[self::UPGRADE_TYPE_MINOR])
        ) {
            unset($response[self::UPGRADE_TYPE_MAJOR]);
        }

        return $response;
    }

    public function getAllEditions(): array
    {
        $params = [];

        return $this->client->retrieve(
            'get_editions',
            $params
        ) ?? [];
    }

    public function registerLicense(string $key, ?int $wave = null): License
    {
        $keyData = $this->retrieveLicense($key, $wave);
        $keyData = $this->prepareLicenseData($keyData);
        $license = $this->entityManager->getRepository(License::class)->findOneBy([
            'moduleId' => $keyData['moduleId'],
            'keyType'  => $keyData['keyType'],
        ]) ?? new License();

        $license->setModuleId($keyData['moduleId']);
        $license->setKeyValue($keyData['keyValue']);
        $license->setKeyType($keyData['keyType']);
        $license->setKeyData($keyData['keyData']);
        $license->setExpiredAt($keyData['expiredAt']);

        if (!$this->entityManager->contains($license)) {
            $this->entityManager->persist($license);
        }

        $this->entityManager->flush();

        return $license;
    }

    public function retrieveLicense(string $key, ?int $wave = null): array
    {
        $params = [
            'key'                    => $key,
            'withModulesVersionInfo' => 1,
            'withCoresVersionInfo'   => 1,
        ];

        if ($wave) {
            $params['wave']       = $wave;
            $params['doRegister'] = 1;
        }

        $result = $this->client->retrieve(
            'check_addon_key',
            $params,
            true
        ) ?: [];

        if (empty($result) || isset($result['message'])) {
            throw new NotFoundHttpException($result['message'] ?? 'License is not found');
        }

        return $result;
    }

    private function prepareLicenseData(array $data): ?array
    {
        $key     = array_key_first($data);
        $keyInfo = isset($data[$key]) ? array_shift($data[$key]) : null;

        if ($keyInfo) {
            $entityFields = [
                'name',
                'author',
                'keyType',
                'keyData',
                'expiredAt',
            ];

            $extraFields = array_filter(
                $keyInfo,
                static fn($key) => !in_array($key, $entityFields),
                ARRAY_FILTER_USE_KEY
            );

            $result = [
                'moduleId'  => $keyInfo['author'] . '-' . $keyInfo['name'],
                'keyValue'  => $key,
                'keyType'   => $keyInfo['keyType'],
                'keyData'   => array_merge($keyInfo['keyData'], $extraFields),
                'expiredAt' => $keyInfo['keyData']['expDate'] ?? 0,
                'wave'      => $keyInfo['keyData']['wave'] ?? null,
            ];
        }

        return $result ?? null;
    }

    public static function explodeVersion(?string $version = ''): array
    {
        [$major1, $major2, $minor, $build] = explode('.', $version);

        return [
            'major' => "{$major1}.{$major2}",
            'minor' => $minor,
            'build' => $build,
        ];
    }

    public static function implodeVersion(array $version): string
    {
        return "{$version['major']}.{$version['minor']}.{$version['build']}";
    }

    /**
     * @param string $author
     * @param string $name
     * @param string $version
     *
     * @return string
     */
    public static function getModuleVersionHash(string $author, string $name, string $version): string
    {
        return md5("{$author}.{$name}.{$version}");
    }

    /**
     * @throws GetTokenDataException
     */
    public function retrieveLicensesByToken(string $token): array
    {
        $params = ['token' => $token];

        $result = $this->client->retrieve(
            'get_token_data',
            $params
        ) ?: [];

        if (!$result) {
            throw GetTokenDataException::fromEmptyResponse();
        }

        if (isset($result['message'])) {
            throw GetTokenDataException::fromResponseWithError($result['message']);
        }

        return $result;
    }

    /**
     * @throws Exception
     */
    public function getInstalledModuleVersionHashes(): array
    {
        $installedModuleVersionHashes = [];

        foreach ($this->moduleRepository->getAllModules() as $module) {
            $moduleId = $module->getModuleId();

            $installedModuleVersionHashes[$moduleId] = self::getModuleVersionHash(
                $module->getAuthor(),
                $module->getName(),
                $module->getVersion(),
            );
        }

        return $installedModuleVersionHashes;
    }

    public function getCoreLicense(): ?License
    {
        return $this->licenseRepository->findCoreLicense();
    }

    private function getCoreEditionData(): array
    {
        $editions    = $this->getAllEditions();
        $coreLicense = $this->getCoreLicense();
        $xcnPlan     = $coreLicense->getKeyData()['xcnPlan'] ?? null;
        $filtered    = array_filter(
            $editions,
            static fn($edition) => (int) $edition['xcnPlan'] === $xcnPlan
        );

        return $filtered ? array_shift($filtered) : [];
    }

    private function getCoreEditionId(): ?int
    {
        $coreEditionId = $this->getCoreEditionData()['xb_product_id'] ?? null;

        if ($coreEditionId !== null) {
            $coreEditionId = (int) $coreEditionId;
        }

        return $coreEditionId;
    }

    public function isModuleIncludedInCoreEdition(array $marketplaceModule): bool
    {
        return match ((int) $marketplaceModule['xcn_plan']) {
            -1 => false,
            0 => (int) $marketplaceModule['price'] === 0,
            1 => (int) $marketplaceModule['edition_state'] === 1,
        };
    }

    public function saveGmvData(string $gmvData): ?array
    {
        $params = ['gmv_data' => $gmvData];

        return $this->client->retrieve(
            'save_gmv_data',
            $params,
            true
        ) ?: [];
    }
}
