<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Logic\InstallRequirements;

final class PhpVersion implements XCartInterface
{
    private string $minPhpVersion;

    private string $maxPhpVersion;

    private array $forbiddenPhpVersions;

    public function __construct(string $minPhpVersion, string $maxPhpVersion = '', array $forbiddenPhpVersions = [])
    {
        $this->minPhpVersion        = $minPhpVersion;
        $this->maxPhpVersion        = $maxPhpVersion;
        $this->forbiddenPhpVersions = $forbiddenPhpVersions;
    }

    public function getServiceName(): string
    {
        return 'php_version';
    }

    public function getTitle(): string
    {
        return 'PHP version';
    }

    public function getLevel(): int
    {
        return static::LEVEL_CRITICAL;
    }

    public function getState(bool $dataCheckerState): int
    {
        return $dataCheckerState ? static::STATE_SUCCESS : static::STATE_FAILURE;
    }

    public function getChecker(): array
    {
        $version = phpversion();

        if (version_compare($version, $this->minPhpVersion) < 0) {
            return [
                false,
                'PHP Version must be at least %minPhpVersion% Your version is %version%',
                [
                    'version'       => $version,
                    'minPhpVersion' => $this->minPhpVersion . '+',
                ],
            ];
        }

        if ($this->maxPhpVersion && version_compare($version, $this->maxPhpVersion) > 0) {
            return [
                false,
                'PHP Version must be not greater than %maxPhpVersion%',
                [
                    '%version%'       => $version,
                    '%maxPhpVersion%' => $this->maxPhpVersion,
                    '%minPhpVersion%' => $this->minPhpVersion . '+',
                ],
            ];
        }

        if ($this->forbiddenPhpVersions && is_array($this->forbiddenPhpVersions)) {
            foreach ($this->forbiddenPhpVersions as $forbiddenPhpVersion) {
                if (
                    version_compare($version, $forbiddenPhpVersion['min']) >= 0
                    && version_compare($version, $forbiddenPhpVersion['max']) <= 0
                ) {
                    return [
                        false,
                        'Unsupported PHP version detected.',
                        [
                            'version'                => $version,
                            'forbiddenPhpVersionMin' => $forbiddenPhpVersion['min'],
                            'forbiddenPhpVersionMax' => $forbiddenPhpVersion['max'],
                        ],
                    ];
                }
            }
        }

        return [true, '', []];
    }
}
