<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Logic\InstallRequirements;

final class PhpMemoryLimit implements XCartInterface
{
    private string $memoryLimitMin;

    private ?array $phpInfo = null;

    public function __construct(string $memoryLimitMin)
    {
        $this->memoryLimitMin = $memoryLimitMin;
    }

    public function getServiceName(): string
    {
        return 'php_memory_limit';
    }

    public function getTitle(): string
    {
        return 'Memory limit';
    }

    public function getLevel(): int
    {
        return static::LEVEL_CRITICAL;
    }

    public function getState(bool $dataCheckerState): int
    {
        return $dataCheckerState ? static::STATE_SUCCESS : static::STATE_FAILURE;
    }

    public function getChecker(): array
    {
        $memoryLimitMin = $this->memoryLimitMin;
        if ($this->isMemoryLimited()) {
            $memoryLimit = @ini_get('memory_limit');

            return [
                $this->convertMemoryLimitToInteger($memoryLimitMin) <= $this->convertMemoryLimitToInteger($memoryLimit),
                'PHP memory_limit option must be at least %memoryLimitMin%',
                [
                    '%memoryLimit%'    => $memoryLimit,
                    '%memoryLimitMin%' => $memoryLimitMin,
                ],
            ];
        } else {
            return [
                true,
                'PHP memory_limit option must be at least %memoryLimitMin%',
                [
                    'memoryLimit'    => 'Unlimited',
                    'memoryLimitMin' => $memoryLimitMin,
                ],
            ];
        }

    }

    private function isMemoryLimited(): bool
    {
        $info = $this->getPhpInfo();

        $unlimited = ($info['no_mem_limit']
                && $info['commands_exists']
                && !function_exists('memory_get_usage')
                && @ini_get('memory_limit') === '')
            || @ini_get('memory_limit') === '-1';

        return !$unlimited;
    }

    private function convertMemoryLimitToInteger($value): int
    {
        $last  = strtolower(substr($value, -1));
        $value = (int) $value;

        switch ($last) {
            case 'k':
                $value *= 1024;
                break;

            case 'm':
                $value *= 1024 * 1024;
                break;

            case 'g':
                $value *= 1024 * 1024 * 1024;
        }

        return (int) $value;
    }

    private function getPhpInfo(): ?array
    {
        if ($this->phpInfo === null) {
            $this->phpInfo = $this->definePhpInfo();
        }

        return $this->phpInfo;
    }

    private function definePhpInfo(): array
    {
        $result = [];

        ob_start();
        phpinfo();
        $rawPhpInfo = ob_get_contents();
        ob_end_clean();

        foreach (explode("\n", $rawPhpInfo) as $line) {
            if (stripos($line, 'command') !== false) {
                $result['commands_exists'] = true; // wtf?
                if (stripos($line, '--enable-memory-limit') !== false) {
                    $result['no_mem_limit'] = false;
                }

                continue;
            }

            if (preg_match('/Phar EXT version.*<\/td><td([^>]*)>([^<]*)/i', $line, $match)) {
                $result['phar_ext_ver'] = $match[2];
            }
        }

        return array_merge(
            [
                'thread_safe'            => false,
                'debug_build'            => false,
                'php_ini_path'           => '',
                'no_mem_limit'           => true,
                'commands_exists'        => false,
                'php_ini_path_forbidden' => false,
                'phar_ext_ver'           => '',
            ],
            $result
        );
    }
}
