<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Logic\InstallRequirements;

use App\Domain\XCart;
use function strtoupper;
use function substr;
use const PHP_OS;

final class FilePermissions implements XCartInterface
{
    public const PERMISSION_DIR  = '777';
    public const PERMISSION_FILE = '666';

    private string $rootPath;

    private bool $isOSWin;

    public function __construct(XCart $XCart)
    {
        $this->rootPath = $XCart->getSourcePath();
        $this->isOSWin  = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';
    }

    private static function getFilePermissionsPaths(): array
    {
        return [
            'var',
            'public',
        ];
    }

    public function getServiceName(): string
    {
        return 'file_permissions';
    }

    public function getTitle(): string
    {
        return 'File permissions';
    }

    public function getLevel(): int
    {
        return static::LEVEL_CRITICAL;
    }

    public function getState(bool $dataCheckerState): int
    {
        return $dataCheckerState ? static::STATE_SUCCESS : static::STATE_FAILURE;
    }

    public function getChecker(): array
    {
        $rootPath             = $this->rootPath;
        $filePermissionsPaths = $this->getFilePermissionsPaths();

        $result = [];
        foreach ($filePermissionsPaths as $path) {
            $result[] = $this->checkPermissions($rootPath . $path);
        }

        $result = call_user_func_array('array_merge', $result);
        if ($result) {
            $perms = [];
            foreach ($result as $file => $perm) {
                if ($this->isOSWin) {
                    $perms[] = $file;
                } else {
                    if (is_dir((string) $file)) {
                        $perms[] = 'chmod 0' . self::PERMISSION_DIR . ' ' . $file;
                        $perms[] = 'find ' . $file . ' -type d -exec chmod 0' . self::PERMISSION_DIR . ' {} \\;';
                        $perms[] = 'find ' . $file . ' -type f -exec chmod 0' . self::PERMISSION_FILE . ' {} \\;';
                    } else {
                        $perms[] = 'chmod 0' . $perm . ' ' . $file;
                    }
                }
                if (count($perms) > 25) {
                    break;
                }
            }

            return [
                false,
                $this->isOSWin
                    ? 'Not enough permissions to run the process'
                    : 'Not enough permissions to run the process. Make sure the permissions are set',
                [
                    '%paths%'     => $result,
                    '%pathsList%' => implode('<br />' . PHP_EOL, $perms),
                ],
            ];
        }

        return [true, '', []];
    }

    private function checkPermissions($path)
    {
        if (strpos(basename($path), '.') !== 0) {
            if (!is_writable($path)) {
                return [$path => is_dir($path) ? self::PERMISSION_DIR : self::PERMISSION_FILE];
            }

            if (!$this->isOSWin && is_dir($path) && !is_executable($path)) {
                return [$path => self::PERMISSION_DIR];
            }
        }

        $result = [];
        if (is_dir($path) && $handle = @opendir($path)) {
            while (($file = readdir($handle)) !== false) {
                // Skip '.', '..', '.htaccess' and other files those names starts from '.'
                if (strpos($file, '.') === 0) {
                    continue;
                }

                $fileRealPath = $path . DIRECTORY_SEPARATOR . $file;

                if (!is_writable($fileRealPath)) {
                    $result[$fileRealPath] = is_dir($fileRealPath) ? self::PERMISSION_DIR : self::PERMISSION_FILE;
                } elseif (is_dir($fileRealPath)) {
                    $result = array_merge($result, $this->checkPermissions($fileRealPath));
                }
            }

            closedir($handle);
        }

        return $result;
    }

}
