<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Logic;

use App\Domain\XCart;
use App\Entity\CheckRequirementsResult;
use App\Exception\CircularDependencyException;

final class CheckRequirements
{
    private XCart $XCart;

    /**
     * @var array Enabled modules list
     */
    private array $enabledModuleIds = [];

    /**
     * @var array The list of modules marked to remove.
     */
    private array $modulesToRemove = [];

    /**
     * @var array Enabled skins list
     */
    private array $sortedEnabledSkinModuleIds = [];

    /**
     * @var array Module dependencies
     */
    private array $dependencies = [];

    /**
     * @var array Module back dependencies
     */
    private array $backDependencies = [];

    /**
     * @var array Module incompatibles
     */
    private array $incompatibles = [];

    /**
     * @var bool
     */
    private bool $isSetDependencyMap = false;

    /**
     * @var array Expected actions
     */
    private array $actions = [];

    private CheckDependencyMap $checkDependencyMap;

    public function __construct(
        XCart $XCart,
        CheckDependencyMap $checkDependencyMap
    ) {
        $this->XCart = $XCart; // for version_compare($this->XCart->getCoreVersion() ...
        $this->checkDependencyMap = $checkDependencyMap;
    }

    /**
     * @param array $dependencies
     *
     * @throws CircularDependencyException
     */
    public function setDependencyMap(array $dependencies): void
    {
        foreach ($dependencies as $directModuleId => $dependency) {
            $this->dependencies[$directModuleId] = $dependency[0] ?? [];

            foreach ($dependency[0] ?? [] as $dependencyId) {
                $this->backDependencies[$dependencyId][] = $directModuleId;
            }

            // cumulative all direct and back incompatibilities
            $this->incompatibles[$directModuleId] = array_unique(array_merge($this->incompatibles[$directModuleId] ?? [], $dependency[1] ?? []));
            foreach ($dependency[1] ?? [] as $incompatibilityId) {
                $this->incompatibles[$incompatibilityId] = array_unique(array_merge($this->incompatibles[$incompatibilityId] ?? [], [$directModuleId]));
            }
        }
        ksort($this->incompatibles);

        if ($cycle = $this->checkDependencyMap->getCycle($this->dependencies)) {
            throw CircularDependencyException::fromDependenciesCheck($cycle);
        }

        $this->isSetDependencyMap = true;
    }

    public function isSetDependencyMap(): bool
    {
        return $this->isSetDependencyMap;
    }

    public function checkActions(): CheckRequirementsResult
    {
        $result = new CheckRequirementsResult();

        foreach ($this->getSortedActions() as $id => $action) {
            if ($this->checkAction($id, $action, $result)) {
                $result->addSuccessActions($id, $action['actionType']);
            } else {
                $result->addFailedActions($id, $action['actionType']);
            }
        }

        return $result;
    }

    private function getSortedActions(): array
    {
        $map = [];
        foreach ($this->actions as $moduleId => $module) {
            $map[$moduleId] = $module['dependsOn'];
        }

        $sortedModules = $this->checkDependencyMap->sort($map);

        $enabled = $disabled = [];
        foreach (array_filter($sortedModules) as $sortedModule) {
            if (in_array($this->actions[$sortedModule]['actionType'], ['disable', 'remove'], true)) {
                $disabled[$sortedModule] = $this->actions[$sortedModule];
            } else {
                $enabled[$sortedModule] = $this->actions[$sortedModule];
            }
        }

        $disabledResult = $this->revertDisabledSortedModules($disabled);

        return array_merge($enabled, $disabledResult);
    }

    private function revertDisabledSortedModules(array $disabled): array
    {
        $disabledResult = [];
        end($disabled);
        while (current($disabled)) {
            $disabledResult[key($disabled)] = current($disabled);
            prev($disabled);
        }

        return $disabledResult;
    }

    /**
     * Check single action (fill appropriate lists)
     */
    public function checkAction(string $id, array $action, CheckRequirementsResult $result): bool
    {
        switch ($action['actionType']) {
            case 'install':
            case 'enable':
                return $this->checkEnableAction($id, $result, $action['isSkin'], $action['minRequiredCoreVersion']);
            case 'disable':
                return $this->checkDisableAction($id, $result, $action['isSkin'], $action['canDisable']);
            case 'remove':
                return $this->checkRemoveAction($id, $result, $action['isSkin'], $action['canDisable']);
            default:
                return false;
        }
    }

    /**
     * Try to enable module
     */
    private function checkEnableAction(string $id, CheckRequirementsResult $result, bool $isSkin, string $minRequiredCoreVersionFull): bool
    {
        if (in_array($id, $result->getToEnable(), true)) {
            // Module $id is already in toEnable list, skip with success
            return true;
        }

        if (in_array($id, $result->getCantEnable(), true)) {
            // Module $id is already in cantEnable list, skip with fail
            return false;
        }

        if ($isSkin && !$this->checkDisableAllSkins($result)) {
            return false;
        }

        if (in_array($id, $result->getToDisable(), true)) {
            if ($isSkin) {
                // If there is the skin $id in the toDisable array,
                // it must be removed from this array
                $result->removeToDisable($id);
            } else {
                // Module $id is already in toDisable list, we can't enable and disable module in same time, fail
                // Add module $id to cantEnable list with reason code
                $result->addCantEnable($id, CheckRequirementsResult::CONFLICT_WITH_DISABLE_REASON);

                return false;
            }
        }

        // todo: check module for availability (local -> enable, remote -> install)

        // check version compatibility if minorRequiredCoreVersion is set for the module in the main.yaml file
        if ($minRequiredCoreVersionFull && !version_compare($this->XCart->getCoreVersion(), $minRequiredCoreVersionFull, '>=')) {
            $result->addCantEnable($id, CheckRequirementsResult::CANT_ENABLE_BY_VERSION_REASON);
            return false;
        }

        // Add modules from toEnable list to enabled list to prevent conflict actions
        $enabled = array_merge($this->enabledModuleIds, $result->getToEnable());
        // Remove toDisable modules from enabled list to prevent conflict actions
        $enabled = array_diff($enabled, $result->getToDisable());
        // Get disabled modules form the dependencies list
        $disabledDependentModules = array_diff($this->dependencies[$id] ?? [], $enabled);

        if ($isSkin) {
            foreach ($disabledDependentModules as $toEnableId) {
                // Try to enable dependency
                if (!$this->checkEnableAction($toEnableId, $result, $isSkin, '')) { // TODO check minRequiredCoreVersion of the dependency
                    // Dependency can't be enabled, fail
                    // Add module $id to cantEnable list with reason code and dependency info
                    $result->addCantEnable($id, CheckRequirementsResult::CANT_ENABLE_BY_DEPENDENCY_REASON, [$toEnableId]);

                    return false;
                }
            }
        } elseif ($disabledDependentModules) {
            // Add module $id to cantEnable list with reason code and dependency info
            $result->addCantEnable(
                $id,
                CheckRequirementsResult::CANT_ENABLE_BY_DEPENDENCY_REASON,
                $disabledDependentModules
            );

            return false;
        }

        // Get enabled modules from the incompatibles list
        $incompatibleEnabledModules = array_intersect($this->incompatibles[$id] ?? [], $enabled);

        if ($incompatibleEnabledModules) {
            // Add module $id to cantEnable list with reason code and incompatible module info
            $result->addCantEnable(
                $id,
                CheckRequirementsResult::CANT_ENABLE_BY_INCOMPATIBLE_REASON,
                $incompatibleEnabledModules
            );

            return false;
        }

        if (
            !$isSkin
            || !in_array($id, $this->sortedEnabledSkinModuleIds, true)
        ) {
            // if skin was already enabled, skip
            // Otherwise add module $id to toEnable list
            $result->addToEnable($id);
        }

        return true;
    }

    /**
     * Try to disable module
     */
    public function checkDisableAction(string $id, CheckRequirementsResult $result, bool $isSkin, bool $canDisable): bool
    {
        if (in_array($id, $result->getToDisable(), true)) {
            // Module $id is already in toDisable list, skip with success
            return true;
        }

        if (in_array($id, $result->getCantDisable(), true)) {
            // Module $id is already in cantDisable list, skip with fail
            return false;
        }

        if (in_array($id, $result->getToEnable(), true)) {
            if ($isSkin) {
                // If there is the skin $id in the toEnable array,
                // it must be removed from this array
                $result->removeToEnable($id);
            } else {
                // Module $id is already in toEnable list, we can't disable and enable module in same time, fail
                // Add module $id to cantDisable list with reason code
                $result->addCantDisable($id, CheckRequirementsResult::CONFLICT_WITH_ENABLE_REASON);

                return false;
            }
        }

        if (!$canDisable) {
            $result->addCantDisable($id, CheckRequirementsResult::CANT_DISABLE_BY_MODULE_PROPERTY_REASON);

            return false;
        }

        // Add modules from toEnable list to enabled list to prevent conflict actions
        $enabled = array_merge($this->enabledModuleIds, $result->getToEnable());
        // Remove toDisable modules from enabled list to prevent conflict actions
        $enabled = array_diff($enabled, $result->getToDisable());

        // Quick fix for XCB-2521
        // FIXME: Remove the line below when XCB-2521 is properly fixed.
        $enabled = array_merge($enabled, $this->getModulesToRemove());

        // Get enabled modules from the back dependencies list
        $backDependenciesEnabledModules = array_intersect($this->backDependencies[$id] ?? [], $enabled);

        if ($backDependenciesEnabledModules) {
            $result->addCantDisable(
                $id,
                CheckRequirementsResult::CANT_DISABLE_BY_BACK_DEPENDENCY_REASON,
                $backDependenciesEnabledModules
            );

            return false;
        }

        // No other reasons, add module $id to toDisable list, success
        $result->addToDisable($id);

        return true;
    }

    /**
     * Try to remove module
     */
    public function checkRemoveAction(string $id, CheckRequirementsResult $result, bool $isSkin, bool $canDisable): bool
    {
        if ($this->checkDisableAction($id, $result, $isSkin, true)) {
            $result->addToRemove($id);

            return true;
        }

        return false;
    }

    /**
     * Try to disable all skins
     */
    public function checkDisableAllSkins(CheckRequirementsResult $result): bool
    {
        $enabledSkinModuleIds = array_diff(
            $this->getSortedEnabledSkinModuleIds(),
            $result->getToDisable()
        );

        foreach ($enabledSkinModuleIds as $moduleId) {
            if (!$this->checkDisableAction($moduleId, $result, true, true)) {
                return false;
            }
        }

        return true;
    }

    public function getEnabledModuleIds(): array
    {
        return $this->enabledModuleIds;
    }

    public function setEnabledModuleIds(array $enabledModuleIds): void
    {
        $this->enabledModuleIds = $enabledModuleIds;
    }

    public function getModulesToRemove(): array
    {
        return $this->modulesToRemove;
    }

    public function setModulesToRemove(array $modulesToRemove): void
    {
        $this->modulesToRemove = $modulesToRemove;
    }

    public function getSortedEnabledSkinModuleIds(): array
    {
        return $this->sortedEnabledSkinModuleIds;
    }

    public function setSortedEnabledSkinModuleIds(array $sortedEnabledSkinModuleIds): void
    {
        $this->sortedEnabledSkinModuleIds = $sortedEnabledSkinModuleIds;
    }

    public function getActions(): array
    {
        return $this->actions;
    }

    public function setActions(array $actions): void
    {
        $this->actions = $actions;
    }

    public function addAction(
        string $actionType,
        string $moduleId,
        bool   $isSkin,
        bool   $canDisable,
        string $minRequiredCoreVersion,
        array  $dependsOn
    ): void {
        $this->actions[$moduleId] = [
            'actionType'             => $actionType,
            'isSkin'                 => $isSkin,
            'canDisable'             => $canDisable,
            'minRequiredCoreVersion' => $minRequiredCoreVersion,
            'dependsOn'              => $dependsOn
        ];
    }
}
