<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\LifetimeHook\Upgrade;

use App\Domain\XCart;
use App\Entity\Scenario;
use App\Repository\ScenarioRepository;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;

final class UpgradeTo550build0
{
    private Filesystem $filesystem;

    private ScenarioRepository $scenarioRepository;

    private string $sourcePath;

    private string $packsPath;

    private ?Scenario $scenario = null;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem,
        ScenarioRepository $scenarioRepository
    ) {
        $this->filesystem         = $filesystem;
        $this->scenarioRepository = $scenarioRepository;
        $this->sourcePath         = $XCart->getSourcePath();
        $this->packsPath          = $XCart->getPacksPath();
    }

    public function onUpgrade(): void
    {
        $toRemove = [
            'files/service/',
            'service/',
            'skins/',
            'upgrade_54_to_55.php',
            'upgrade_54_to_55/',
        ];

        foreach (Finder::create()->directories()->in($this->sourcePath . 'classes/XLite/Module')->depth(0) as $dir) {
            $toRemove[] = $this->filesystem->makePathRelative($dir->getPathname(), $this->sourcePath);
        }

        foreach ($toRemove as $path) {
            $this->backup($path);
            $this->filesystem->remove($this->sourcePath . $path);
        }
    }

    private function backup($path): void
    {
        $backupPath = $this->packsPath . $this->getScenario()->getId() . '/';

        $sourcePath = $this->sourcePath . $path;

        if (is_dir($sourcePath)) {
            $this->filesystem->mirror($sourcePath, $backupPath . $path);
        } elseif (is_file($sourcePath)) {
            $this->filesystem->copy($sourcePath, $backupPath . $path);
        }
    }

    private function getScenario(): Scenario
    {
        if ($this->scenario === null) {
            /** @var Scenario $scenario */
            $this->scenario = $this->scenarioRepository->findLastScenario();
        }

        return $this->scenario;
    }
}
