<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Filter;

use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\AbstractContextAwareFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Util\QueryNameGeneratorInterface;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Serializer\NameConverter\NameConverterInterface;
use ApiPlatform\Core\Exception\InvalidArgumentException;
use App\Traits\JsonFilterTrait;

class ModuleSearchFilter extends AbstractContextAwareFilter
{
    use JsonFilterTrait;

    private $searchParameterName;

    public function __construct(
        ManagerRegistry $managerRegistry,
        ?RequestStack $requestStack = null,
        LoggerInterface $logger = null,
        array $properties = null,
        NameConverterInterface $nameConverter = null,
        string $searchParameterName = 'search'
    ) {
        parent::__construct($managerRegistry, $requestStack, $logger, $properties, $nameConverter);

        $this->searchParameterName = $searchParameterName;
    }

    protected function filterProperty(
        string $property,
        $value,
        QueryBuilder $queryBuilder,
        QueryNameGeneratorInterface $queryNameGenerator,
        string $resourceClass,
        string $operationName = null,
        array $context = []
    ) {
        if (
            !$this->isJsonProperty($property)
            && (
                $value === null
                || $property !== $this->searchParameterName
            )
        ) {
            return;
        }

        $words = explode(' ', $value);

        foreach ($words as $word) {
            if (empty($word)) {
                continue;
            }

            $alias = $queryBuilder->getRootAliases()[0];

            $this->addWhere($alias, $queryBuilder, $word, $queryNameGenerator->generateParameterName($property));
        }
    }

    private function addWhere($alias, $queryBuilder, $word, $parameterName)
    {
        $orExp = $queryBuilder->expr()->orX();
        foreach ($this->getProperties() as $prop => $ignored) {
            [$field, $path] = $this->splitJsonProperty($prop);

            $orExp->add(
                $queryBuilder->expr()->like(
                    'LOWER(JSON_EXTRACT(' . $alias . '.' . $field . ', \'$.' . $path . '\'))', ':' . $parameterName
                )
            );
        }

        $queryBuilder
            ->andWhere('(' . $orExp . ')')
            ->setParameter($parameterName, '%' . strtolower($word) . '%');
    }

    public function getDescription(string $resourceClass): array
    {
        $props = $this->getProperties();

        return [
            $this->searchParameterName => [
                'property' => implode(', ', array_keys($props)),
                'type'     => 'string',
                'required' => false,
            ],
        ];
    }
}
