<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Filter;

use ApiPlatform\Core\Bridge\Doctrine\Common\Filter\OrderFilterTrait;
use App\Traits\JsonFilterTrait;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\OrderFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Util\QueryNameGeneratorInterface;
use Doctrine\ORM\QueryBuilder;

/**
 * This filter allows ordering of records by value in JSON field by path
 * The syntax of property name is [EntityFieldName]->[JSONPath]
 * EntityFieldName can be a field of nested entity
 */
final class JsonSupportOrderFilter extends OrderFilter
{
    use OrderFilterTrait;
    use JsonFilterTrait;

    protected function isPropertyMapped(string $property, string $resourceClass, bool $allowAssociation = false): bool
    {
        if ($this->isJsonProperty($property)) {
            [$property,] = $this->splitJsonProperty($property);
        }

        return parent::isPropertyMapped($property, $resourceClass, $allowAssociation);
    }

    protected function filterProperty(string $property, $direction, QueryBuilder $queryBuilder, QueryNameGeneratorInterface $queryNameGenerator, string $resourceClass, string $operationName = null): void
    {
        if (!$this->isJsonProperty($property)) {
            parent::filterProperty($property, $direction, $queryBuilder, $queryNameGenerator, $resourceClass, $operationName);

            return;
        }

        $alias = $queryBuilder->getRootAliases()[0];

        [$field, $path] = $this->splitJsonProperty($property);

        $orderingField = sprintf('_%s_%s_%s_ordering_field', $alias, str_replace('.', '_', $field), str_replace('.', '_', $path));

        $queryBuilder->addSelect(sprintf('LOWER(JSON_EXTRACT(%s.%s, \'$.%s\')) AS HIDDEN %s', $alias, $field, $path, $orderingField));

        $direction = $this->normalizeValue($direction, $property);
        if (null === $direction) {
            return;
        }

        if (null !== $nullsComparison = $this->properties[$property]['nulls_comparison'] ?? null) {
            $nullsDirection = self::NULLS_DIRECTION_MAP[$nullsComparison][$direction];

            $nullRankHiddenField = sprintf('_%s_%s_%s_null_rank', $alias, str_replace('.', '_', $field), str_replace('.', '_', $path));

            $queryBuilder->addSelect(sprintf('CASE WHEN %s IS NULL THEN 0 ELSE 1 END AS HIDDEN %s', $orderingField, $nullRankHiddenField));
            $queryBuilder->addOrderBy($nullRankHiddenField, $nullsDirection);
        }

        $queryBuilder->addOrderBy($orderingField, $direction);
    }
}
