<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Filter;

use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\AbstractContextAwareFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\BooleanFilter;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Util\QueryNameGeneratorInterface;
use App\Marketplace\Marketplace;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\PropertyInfo\Type;
use Symfony\Component\Serializer\NameConverter\NameConverterInterface;

/**
 * This filter dynamically set custom flag for using custom filter as db value
 */
final class CustomModulesFilter extends BooleanFilter
{
    private Marketplace $marketplace;
    private ModuleRepository $moduleRepository;
    private EntityManagerInterface $entityManager;

    public function __construct(
        ManagerRegistry        $managerRegistry,
        ?RequestStack          $requestStack = null,
        LoggerInterface        $logger = null,
        array                  $properties = null,
        NameConverterInterface $nameConverter = null,
        Marketplace            $marketplace = null,
        ModuleRepository       $moduleRepository = null,
        EntityManagerInterface $entityManager = null
    ) {
        parent::__construct($managerRegistry, $requestStack, $logger, $properties, $nameConverter);

        $this->marketplace      = $marketplace;
        $this->moduleRepository = $moduleRepository;
        $this->entityManager    = $entityManager;
    }

    protected function filterProperty(string $property, $value, QueryBuilder $queryBuilder, QueryNameGeneratorInterface $queryNameGenerator, string $resourceClass, string $operationName = null): void
    {
        if (
            $property === 'isCustom'
            && in_array($value, ['true', 'false'])
        ) {
            $this->moduleRepository->resetCustomFlag();
            $this->fillModulesCustomFlag();

            parent::filterProperty($property, $value, $queryBuilder, $queryNameGenerator, $resourceClass, $operationName);
        }

        return;
    }

    public function getDescription(string $resourceClass): array
    {
        $description['isCustom'] = [
            'property' => 'isCustom',
            'type' => Type::BUILTIN_TYPE_BOOL,
            'required' => false,
            'swagger' => [
                'description' => 'If true, only custom modules will be provided'
            ],
        ];

        return $description;
    }

    protected function fillModulesCustomFlag(): void
    {
        $all = $this->moduleRepository->getAllModules();
        foreach ($all as $module) {
            $dynamicData = $this->marketplace->getModule($module->getAuthor(), $module->getName());
            if (empty($dynamicData)) {
                $module->setIsCustom(true);
            }
        }

        $this->entityManager->flush();
    }
}
