<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Exception;

use App\Entity\CheckRequirementsResult;
use Exception;

final class BuildException extends Exception
{
    protected string $lastReasonCode = '';

    public static function fromCyclesInDependencies(CircularDependencyException $e): self
    {
        return new self($e->getMessage(), $e->getCode(), $e);
    }

    public static function fromNotAllowedModuleAction(array $failedActions, array $reasons): self
    {
        $translationMap[CheckRequirementsResult::CANT_DISABLE_BY_BACK_DEPENDENCY_REASON] = 'Cannot be disabled. The module is required by: ';
        $translationMap[CheckRequirementsResult::CANT_DISABLE_BY_MODULE_PROPERTY_REASON] = 'The module may not be disabled due to the limitations of the module architecture.';
        $translationMap[CheckRequirementsResult::CANT_ENABLE_BY_DEPENDENCY_REASON]       = 'To enable this addon, the following addon(s) must be enabled: ';
        $translationMap[CheckRequirementsResult::CANT_ENABLE_BY_INCOMPATIBLE_REASON]     = 'To enable this addon, the following addon(s) must be disabled: ';
        $translationMap[CheckRequirementsResult::CANT_ENABLE_BY_VERSION_REASON]          = '"minorRequiredCoreVersion" directive from the main.yaml file is greater than the current X-Cart core version';
        $translationMap[CheckRequirementsResult::CONFLICT_WITH_DISABLE_REASON]           = 'The module is already in the toDisable list, we cannot enable and disable the module at the same time';
        $translationMap[CheckRequirementsResult::CONFLICT_WITH_ENABLE_REASON]            = 'The module is already in the toEnable list, we cannot disable and enable the module at the same time';

        $resultMessage = '';
        $lastErrorCode = '';
        foreach ($failedActions as $moduleName => $action) {
            $resultMessage .= "$action $moduleName.";
            if ($reasons) {
                $resultMessage .= ' Reasons: ';

                foreach ($reasons as $reason) {
                    if (
                        is_array($reason[0] ?? null)
                        && is_string($reason[0][0] ?? null)
                        && isset($translationMap[$reason[0][0]])
                    ) {
                        $lastErrorCode = $reason[0][0];
                        $resultMessage .= $translationMap[$reason[0][0]] . implode(',', $reason[0][1] ?? []);
                    } else {
                        $resultMessage .= print_r($reason, true);
                    }
                }
            }
            $resultMessage .= "\n";
        }

        $exception = new self(sprintf('Some actions are not allowed: %s', $resultMessage));
        $exception->setLastReasonCode($lastErrorCode);

        return $exception;
    }

    public function getLastReasonCode(): string
    {
        return $this->lastReasonCode;
    }

    protected function setLastReasonCode(string $code): void
    {
        $this->lastReasonCode = $code;
    }
}
