<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\EventSubscriber;

use App\Entity\License;
use App\DTO\Output\LicenseOutput;
use App\Marketplace\Marketplace;
use ApiPlatform\Core\EventListener\EventPriorities;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ViewEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Mime\Email;

use ApiPlatform\Core\Bridge\Doctrine\Orm\QueryAwareInterface;

final class UpdateLicenseSubscriber implements EventSubscriberInterface
{
    const TTL = 86400;

    private Marketplace $marketplace;

    public function __construct(Marketplace $marketplace)
    {
        $this->marketplace = $marketplace;
    }

    public static function getSubscribedEvents()
    {
        return [
            KernelEvents::VIEW => ['updateLicenseEvent', EventPriorities::PRE_SERIALIZE],
        ];
    }

    public function updateLicenseEvent(ViewEvent $event): void
    {
        $result = $event->getControllerResult();

        if ($result instanceof QueryAwareInterface) {
            foreach ($result->getIterator() as $item) {
                if ($item instanceof License) {
                    $this->updateLicense($item);
                }
            }
        } elseif ($result instanceof License) {
            $this->updateLicense($result);

        } elseif ($result instanceof LicenseOutput) {
            $this->updateLicenseByLicenseOutput($result);
        }
    }

    private function updateLicense(License $license): void
    {
        if (
            $license->getUpdatedAt() < (time() - self::TTL)
            && !$license->isTrial()
        ) {
            $this->marketplace->registerLicense($license->getKeyValue());
        }
    }

    private function updateLicenseByLicenseOutput(LicenseOutput $license): void
    {
        if (
            $license->updatedAt < (time() - self::TTL)
            && !$license->isTrial
        ) {
            $this->marketplace->registerLicense($license->keyValue);
        }
    }
}
