<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\EventListener;

use App\Entity\Scenario;
use App\Event\RunnerEvent;
use App\Exception\ClearServiceToolCacheException;
use App\Operation\Build\RebuildFlag;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Workflow\WorkflowInterface;
use Throwable;

class RunnerListener
{
    private RebuildFlag $rebuildFlag;

    private WorkflowInterface $rebuildStateMachine;

    private EntityManagerInterface $entityManager;

    public function __construct(
        RebuildFlag $rebuildFlag,
        WorkflowInterface $rebuildStateMachine,
        EntityManagerInterface $entityManager
    ) {
        $this->rebuildFlag         = $rebuildFlag;
        $this->rebuildStateMachine = $rebuildStateMachine;
        $this->entityManager       = $entityManager;
    }

    public function start(RunnerEvent $event): void
    {
        $scenario = $event->getScenario();
        $scenarioMetadata = $scenario->getMetaData();

        $scenarioMetadata['runnerSteps'] = [
            'current' => 0,
            'total' => count($this->rebuildStateMachine->getDefinition()->getPlaces()) - 1, // exclude draft place
        ];

        $scenario->setMetaData($scenarioMetadata);
        $scenario->setState(Scenario::STATE_IN_PROGRESS);

        $this->rebuildFlag->create((string) $scenario->getId());
    }

    /**
     * @noinspection PhpRedundantCatchClauseInspection
     * @throws ClearServiceToolCacheException
     * @throws Throwable
     */
    public function runNextStep(RunnerEvent $event): void
    {
        $scenario = $event->getScenario();

        if ($currentTransition = $this->getNextTransition($scenario)) {
            $output = $event->getOutput();

            $this->rebuildStateMachine->apply(
                $scenario,
                $currentTransition,
                ['output' => $output]
            );

            $scenarioMetadata = $scenario->getMetaData();
            $scenarioMetadata['runnerSteps']['current']++;
            $scenario->setMetaData($scenarioMetadata);

            $scenario->setState(Scenario::STATE_IN_PROGRESS);

            $scenario->addProgressMessages(
                $output->getBufferData()
            );

            $output->clearBuffer();

            $this->entityManager->getConnection()->close();
            $this->entityManager->getConnection()->connect();

            $this->entityManager->flush();

            // After move_packs step a new runtime should start with a new up-to-date cache
            if (
                $currentTransition === 'move_packs'
                && $scenario->getType() === Scenario::TYPE_UPGRADE
            ) {
                throw new ClearServiceToolCacheException();
            }
        }
    }

    public function abort(RunnerEvent $event): void
    {
        $scenario = $event->getScenario();
        $scenario->setState(Scenario::STATE_ERROR);

        $this->entityManager->flush();

        $this->rebuildFlag->remove();
    }

    public function finish(RunnerEvent $event): void
    {
        $scenario = $event->getScenario();
        $output   = $event->getOutput();

        $scenario->setState(Scenario::STATE_DONE);
        $output->addEndMessage('Done');

        $scenario->addProgressMessages(
            $output->getBufferData()
        );

        $this->entityManager->flush();

        $this->rebuildFlag->remove();
    }

    private function getNextTransition(Scenario $scenario): string
    {
        $transition = current($this->rebuildStateMachine->getEnabledTransitions($scenario));

        return $transition ? $transition->getName() : '';
    }
}
