<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use App\Controller\CreateNonPersistentScenario;
use App\Controller\DownloadPacksController;
use App\Controller\SetUpgradeEntriesController;
use App\Controller\StartUpgradeScenarioController;
use App\DataPersister\NonPersistentScenarioDataPersister;
use App\DataProvider\RebuildScenarioDataProvider;
use App\DTO\Input\NonPersistentScenarioInput;
use App\DTO\Input\ScenarioInput;
use App\DTO\Input\UpgradeScenarioInput;
use App\DTO\Output\ScenarioDownloadPacksProgressOutput;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\UuidV4;

/**
 * @ORM\Entity
 * @ORM\Table(
 *     name="service_scenarios"
 * )
 *
 * @ApiResource(
 *     paginationEnabled=false,
 *     collectionOperations={
 *          "get",
 *          "post"={
 *              "method"="POST",
 *              "input"=ScenarioInput::class
 *          },
 *          "post_upgrade_scenario"={
 *              "method"="POST",
 *              "input"=UpgradeScenarioInput::class,
 *              "path"="scenarios/upgrade",
 *              "openapi_context"={
 *                  "summary"="Creates upgrade scenario"
 *              }
 *          },
 *          "post_non_persistent_scenario"={
 *              "method"="POST",
 *              "input"=NonPersistentScenarioInput::class,
 *              "persister"=NonPersistentScenarioDataPersister::class,
 *              "path"="scenarios/nonpersistent",
 *              "openapi_context"={
 *                  "summary"="Creates non-persistent scenario with transitions"
 *              }
 *          }
 *     },
 *     itemOperations={
 *          "get",
 *          RebuildScenarioDataProvider::OPERATION_NAME={
 *              "method"="GET",
 *              "path"="scenarios/{id}/rebuild",
 *              "openapi_context"={
 *                  "summary"="Executes the next step of the rebuild and retrieves the scenario",
 *                  "parameters"={
 *                      {
 *                          "name"="id",
 *                          "in"="path",
 *                          "required"=true,
 *                          "schema"={"type"="string"},
 *                          "description"="Scenario id"
 *                      }
 *                  }
 *              }
 *          },
 *          "set_upgrade_entries"={
 *              "method"="PUT",
 *              "deserialize"=false,
 *              "path"="scenarios/upgrade/{id}/upgrade_entries",
 *              "controller"=SetUpgradeEntriesController::class,
 *              "openapi_context"={
 *                  "summary"="Set upgradeEntries to the upgrade scenario metadata",
 *              }
 *          },
 *          "download_packs"={
 *              "method"="PUT",
 *              "output"=ScenarioDownloadPacksProgressOutput::class,
 *              "deserialize"=false,
 *              "path"="scenarios/upgrade/{id}/download_packs",
 *              "controller"=DownloadPacksController::class,
 *              "openapi_context"={
 *                  "summary"="Download packs for the upgrade scenario",
 *              }
 *          },
 *          "start_upgrade_scenario"={
 *              "method"="PUT",
 *              "deserialize"=false,
 *              "path"="scenarios/upgrade/{id}/start",
 *              "controller"=StartUpgradeScenarioController::class,
 *              "openapi_context"={
 *                  "summary"="Start upgrade scenario",
 *              }
 *          },
 *     }
 * )
 */
class Scenario
{
    public const STATE_INIT        = 'init';
    public const STATE_IN_PROGRESS = 'in_progress';
    public const STATE_ERROR       = 'error';
    public const STATE_DONE        = 'done';

    public const TYPE_DRAFT            = 'draft';
    public const TYPE_INSTALL          = 'install';
    public const TYPE_REBUILD          = 'rebuild';
    public const TYPE_UPGRADE          = 'upgrade';
    public const TYPE_UPGRADE_54_TO_55 = 'upgrade54to55';

    /**
     * @ORM\Id
     * @ORM\Column(type="uuid", unique=true)
     */
    private UuidV4 $id;

    /**
     * Workflow places (see config/packages/workflow.yaml)
     *
     * @ORM\Column(type="text", length=32)
     */
    private string $place;

    /**
     * Possible values: init, in-progress, done
     *
     * @ORM\Column(type="text", length=32)
     */
    private string $state;

    /**
     * Possible values: install, rebuild, update, etc
     *
     * @ORM\Column(type="text", length=32)
     */
    private string $type;

    /**
     * Scenario creation timestamp
     *
     * @ORM\Column(type="integer")
     */
    private int $date;

    /**
     * @ORM\Column(type="json")
     */
    private array $transitions = [];

    /**
     * @ORM\Column(type="json")
     */
    private array $metaData = [];

    /**
     * @ORM\Column(type="json")
     */
    private array $stepsData = [];

    /**
     * @ORM\Column(type="json")
     */
    private array $progressMessages = [];

    public function __construct()
    {
        $this->id    = UuidV4::v4();
        $this->place = 'draft';
        $this->type  = static::TYPE_DRAFT;
        $this->state = static::STATE_INIT;

        $this->setCurrentDate();
    }

    public function isDone(): bool
    {
        return $this->place === 'done';
    }

    public function getId(): UuidV4
    {
        return $this->id;
    }

    public function setId(UuidV4 $id): void
    {
        $this->id = $id;
    }

    public function getPlace(): string
    {
        return $this->place;
    }

    public function setPlace(string $place): void
    {
        $this->place = $place;
    }

    public function getState(): string
    {
        return $this->state;
    }

    public function setState(string $state): void
    {
        $this->state = $state;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function setType(string $type): void
    {
        $this->type = $type;
    }

    public function getTransitions(): array
    {
        return $this->transitions;
    }

    public function setTransitions(array $transitions): void
    {
        $this->transitions = $transitions;
    }

    public function getMetaData(): array
    {
        return $this->metaData;
    }

    public function setMetaData(array $metaData): void
    {
        $this->metaData = $metaData;
    }

    public function hasStepData(string $step): bool
    {
        return isset($this->stepsData[$step]);
    }

    public function getStepData(string $step): array
    {
        return $this->stepsData[$step] ?? [];
    }

    public function setStepData(string $step, $data): void
    {
        $this->stepsData[$step] = $data;
    }

    /**
     * @return int
     */
    public function getDate(): int
    {
        return $this->date;
    }

    /**
     * @param int $date
     */
    public function setDate(int $date): void
    {
        $this->date = $date;
    }

    public function setCurrentDate(): void
    {
        $time = new \DateTime('now');

        $this->date = $time->getTimestamp();
    }

    public function getProgressMessages(): array
    {
        return $this->progressMessages;
    }

    public function setProgressMessages(array $progressMessages): void
    {
        $this->progressMessages = $progressMessages;
    }

    public function addProgressMessages(array $progressMessages): void
    {
        $this->progressMessages = array_merge($this->progressMessages, $progressMessages);
    }
}
