<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Entity;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiResource;
use ApiPlatform\Core\Bridge\Doctrine\Orm\Filter\SearchFilter;
use App\DataProvider\GdprModuleDataProvider;
use App\DataProvider\MarketplaceDisallowedModulesDataProvider;
use App\DataProvider\ModuleByModuleIdItemDataProvider;
use App\DTO\Input\ModuleInput;
use App\DTO\Output\MarketplaceModulesOutput;
use App\DTO\Output\ModuleOutput;
use App\Filter\CustomModulesFilter;
use App\Filter\JsonSupportOrderFilter;
use App\Filter\ModuleSearchFilter;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\UuidV4;

/**
 * @ORM\Entity
 * @ORM\Table(
 *     name="service_module",
 *     uniqueConstraints={
 *         @ORM\UniqueConstraint (name="module", columns={"author","name","source","version"})
 *     }
 * )
 *
 * It is required to leave default GET item operation, otherwise it is required to implement Iri generator
 * @ApiResource(
 *     input=ModuleInput::class,
 *     output=ModuleOutput::class,
 *     attributes={
 *      "pagination_client_items_per_page"=true
 *     },
 *     collectionOperations={
 *      "post"={
 *          "openapi_context"={
 *              "summary"="Download and register module from Marketplace by moduleId",
 *          }
 *      },
 *      "get"={
 *          "maximum_items_per_page"=100,
 *          "openapi_context"={
 *              "parameters"={
 *                  {
 *                      "name"="transitions",
 *                      "in"="query",
 *                      "required"=true,
 *                      "description"="Transition list (for calculating the available actions for the module)",
 *                      "schema"={"type"="array"},
 *                      "items"={
 *                          "type"="object",
 *                          "additionalProperties"={"type"="string"}
 *                      },
 *                      "example"={
 *                          "CDev-Sale"="enable",
 *                          "XC-FreeShipping"="disable"
 *                      }
 *                  }
 *              }
 *          }
 *      },
 *      GdprModuleDataProvider::OPERATION_NAME={
 *          "method"="get",
 *          "path"="modules/gdpr",
 *          "pagination_enabled"=false,
 *          "openapi_context"={
 *              "summary"="Retrieves GDPR modules."
 *          }
 *      },
 *      MarketplaceDisallowedModulesDataProvider::OPERATION_NAME={
 *          "method"="get",
 *          "path"="modules/disallowed",
 *          "pagination_enabled"=false,
 *          "output"=MarketplaceModulesOutput::class
 *      }
 *     },
 *     itemOperations={
 *      "get",
 *      ModuleByModuleIdItemDataProvider::OPERATION_NAME={
 *          "method"="GET",
 *          "path"="modules/byModuleId/{moduleId}",
 *          "identifiers"={"moduleId"},
 *          "openapi_context"={
 *              "parameters"={
 *                  {
 *                      "name"="moduleId",
 *                      "in"="path",
 *                      "required"=true,
 *                      "schema"={"type"="string"},
 *                      "description"="String with author code and name code separated by dash (-)"
 *                  }
 *              }
 *          }
 *      }
 *     }
 * )
 *
 * Available values for state property: Module::STATE_*
 * @ApiFilter(SearchFilter::class, properties={"state"})
 * @ApiFilter(JsonSupportOrderFilter::class, properties={"metaData->moduleName", "enabledDate"=JsonSupportOrderFilter::DIRECTION_DESC})
 * @ApiFilter(ModuleSearchFilter::class, properties={"metaData->moduleName", "metaData->description"})
 * @ApiFilter(CustomModulesFilter::class, properties={"isCustom"})
 */
class Module
{
    public const SOURCE_LOCAL  = 'local';
    public const SOURCE_MARKET = 'market';

    public const STATE_NOT_INSTALLED = 'not-installed';
    public const STATE_INSTALLED     = 'installed';
    public const STATE_ENABLED       = 'enabled';
    public const STATE_REMOVED       = 'removed';

    /**
     * @ORM\Id
     * @ORM\Column(type="uuid", unique=true)
     */
    private UuidV4 $id;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $moduleId = '';

    /**
     * @ORM\Column(type="string", length=32)
     */
    private string $author = '';

    /**
     * @ORM\Column(type="string", length=128)
     */
    private string $name = '';

    /**
     * @ORM\Column(type="string", length=16)
     */
    private string $source = self::SOURCE_LOCAL;

    /**
     * @ORM\Column(type="string", length=16)
     */
    private string $version = '';

    /**
     * @ORM\Column(type="string", length=16)
     */
    private string $lastEnabledVersion = '';

    /**
     * @ORM\Column(type="string")
     */
    private string $state = self::STATE_NOT_INSTALLED;

    /**
     * @ORM\Column(type="integer")
     */
    private int $enabledDate = 0;

    /**
     * @ORM\Column(type="json")
     */
    private array $metaData = [];

    /**
     * @ORM\Column(type="json")
     */
    private array $structure = [];

    /**
     * @ORM\Column(type="boolean")
     */
    protected bool $isCustom = false;

    /**
     * @ORM\Column(type="boolean")
     */
    protected bool $hasLocalFiles = true;

    public function __construct()
    {
        $this->id = UuidV4::v4();
    }

    public function isEnabled(): bool
    {
        return $this->state === static::STATE_ENABLED;
    }

    public function isSkin(): bool
    {
        return $this->metaData['type'] === 'skin';
    }

    public function isCustom(): bool
    {
        return $this->isCustom;
    }

    public function getId(): UuidV4
    {
        return $this->id;
    }

    public function setId(UuidV4 $id): void
    {
        $this->id = $id;
    }

    public function getModuleId(): string
    {
        return $this->moduleId;
    }

    public function setModuleId(string $moduleId): void
    {
        $this->moduleId = $moduleId;
    }

    public function getAuthor(): string
    {
        return $this->author;
    }

    public function setAuthor(string $author): void
    {
        $this->author = $author;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function setName(string $name): void
    {
        $this->name = $name;
    }

    public function getSource(): string
    {
        return $this->source;
    }

    public function setSource(string $source): void
    {
        $this->source = $source;
    }

    public function getVersion(): string
    {
        return $this->version;
    }

    public function setVersion(string $version): void
    {
        $this->version = $version;
    }

    public function getLastEnabledVersion(): string
    {
        return $this->lastEnabledVersion;
    }

    public function setLastEnabledVersion(string $version): void
    {
        $this->lastEnabledVersion = $version;
    }

    public function getState(): string
    {
        return $this->state;
    }

    public function setState(string $state): void
    {
        if (!in_array($state, [self::STATE_NOT_INSTALLED, self::STATE_INSTALLED, self::STATE_ENABLED, self::STATE_REMOVED], true)) {
            $state = self::STATE_NOT_INSTALLED;
        }

        $this->state = $state;
    }

    public function getEnabledDate(): int
    {
        return $this->enabledDate;
    }

    public function setEnabledDate(int $enabledDate): void
    {
        $this->enabledDate = $enabledDate;
    }

    public function getMetaData(): array
    {
        return $this->metaData;
    }

    public function setMetaData(array $metaData): void
    {
        $this->metaData = $metaData;
    }

    public function getStructure(): array
    {
        return $this->structure;
    }

    public function setStructure(array $structure): void
    {
        $this->structure = $structure;
    }

    public function setIsCustom(bool $isCustom): void
    {
        $this->isCustom = $isCustom;
    }

    public function hasLocalFiles(): bool
    {
        return $this->hasLocalFiles;
    }

    /**
     * @param bool $hasLocalFiles
     */
    public function setHasLocalFiles(bool $hasLocalFiles): void
    {
        $this->hasLocalFiles = $hasLocalFiles;
    }
}
