<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Entity;

use DateTime;
use App\DTO\Input\LicenseInput;
use App\DTO\Output\LicenseOutput;
use App\DTO\Output\LicenseWithModuleInfoOutput;
use App\DataProvider\CoreLicenseDataProvider;
use Doctrine\ORM\Mapping as ORM;
use ApiPlatform\Core\Annotation\ApiResource;

/**
 * @ORM\Entity
 * @ORM\Table(
 *     name="service_licenses"
 * )
 * @ORM\HasLifecycleCallbacks
 * @ApiResource(
 *     input=LicenseInput::class,
 *     output=LicenseOutput::class,
 *     itemOperations={
 *          "get",
 *          CoreLicenseDataProvider::OPERATION_NAME={
 *              "method"="get",
 *              "path"="licenses/core",
 *              "openapi_context"={
 *                  "summary"="Get Core license data",
 *              }
 *          }
 *     },
 *     collectionOperations={
 *          "get",
 *          "post"={
 *              "openapi_context"={
 *                  "summary"="Register license key",
 *              }
 *          }
 *     }
 * )
 */
class License
{
    public const CORE_KEY_TYPE    = 2;
    public const FREE_LICENSE_KEY = 'XC_TRIAL';
    public const TRIAL_PERIOD     = 2592000;

    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(type="integer")
     */
    private int $id;

    /**
     * @ORM\Column(type="string", length=64)
     */
    private string $moduleId;

    /**
     * @ORM\Column(type="string", length=64, unique=true)
     */
    private string $keyValue;

    /**
     * @ORM\Column(type="integer")
     */
    private int $keyType;

    /**
     * @ORM\Column(type="json")
     */
    private array $keyData = [];

    /**
     * @ORM\Column(type="integer")
     */
    private int $registeredAt;

    /**
     * @ORM\Column(type="integer")
     */
    private int $expiredAt;

    /**
     * @ORM\Column(type="integer")
     */
    private int $updatedAt;

    public function getId(): int
    {
        return $this->id;
    }

    public function getModuleId(): string
    {
        return $this->moduleId;
    }

    public function setModuleId(string $moduleId): void
    {
        $this->moduleId = $moduleId;
    }

    public function getKeyValue(): string
    {
        return $this->keyValue;
    }

    public function setKeyValue(string $keyValue): void
    {
        $this->keyValue = $keyValue;
    }

    public function getKeyType(): int
    {
        return $this->keyType;
    }

    public function setKeyType(int $keyType): void
    {
        $this->keyType = $keyType;
    }

    public function getKeyData(): array
    {
        return $this->keyData;
    }

    public function setKeyData(array $keyData): void
    {
        $this->keyData = $keyData;
    }

    public function getRegisteredAt(): int
    {
        return $this->registeredAt;
    }

    /**
     * @ORM\PrePersist
     */
    public function setRegisteredAt(): void
    {
        $this->registeredAt = (new DateTime('now'))->getTimestamp();
    }

    public function getExpiredAt(): int
    {
        return $this->expiredAt;
    }

    public function setExpiredAt(int $expiredAt): void
    {
        $this->expiredAt = $expiredAt;
    }

    public function getUpdatedAt(): int
    {
        return $this->updatedAt;
    }

    /**
     * @ORM\PreUpdate
     * @ORM\PrePersist
     */
    public function setUpdatedAt(): void
    {
        $this->updatedAt = (new DateTime('now'))->getTimestamp();
    }

    public function getEdition(): string
    {
        return $this->getKeyData()['editionName'] ?? 'Trial';
    }

    public function getReadableName(): ?string
    {
        if ($this->isCoreLicense()) {
            return 'Core';
        }

        return $this->getModule()['readableName'] ?? null;
    }

    public function getModule(): ?array
    {
        $modules = $this->getKeyData()['modules'] ?? [];

        // Array $this->getKeyData()['modules'] always has only one element
        return array_pop($modules);
    }

    public function getProlongKey(): ?int
    {
        return $this->getKeyData()['prolongKey'] ?? null;
    }

    public function getMessage(): ?string
    {
        return $this->getKeyData()['message'] ?? null;
    }

    public function isCoreLicense(): bool
    {
        return $this->getKeyType() === self::CORE_KEY_TYPE;
    }

    public function isExpired(): bool
    {
        return $this->expiredAt < time() && $this->expiredAt !== 0;
    }

    public function isTrial(): bool
    {
        return $this->getKeyValue() === self::FREE_LICENSE_KEY;
    }

    public function isTrialExpired(): bool
    {
        return $this->isTrial() && $this->isExpired();
    }
}
