<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Entity;

use Symfony\Component\Yaml\Exception\ParseException;
use Symfony\Component\Yaml\Yaml;

final class ConfigurationFile
{
    private const FIELDS = ['EnabledModules', 'DisabledModules', 'ExternalModules', 'Data', 'DemoData'];
    private const OPTIONS = ['Lang'];

    private string $filePath;

    private array $options;

    private array $additionalMutationFiles = [];

    public function __construct(string $filePath, array $options = [])
    {
        $this->filePath = $filePath;
        $this->options  = $options;
    }

    public function getConfiguration(): array
    {
        return $this->applyAdditionalMutations($this->applyInternalMutations($this->loadFromFile($this->filePath)));
    }

    public function getAdditionalMutationFiles(): array
    {
        return $this->additionalMutationFiles;
    }

    public function setAdditionalMutationFiles(array $additionalMutationFiles): void
    {
        $this->additionalMutationFiles = $additionalMutationFiles;
    }

    /**
     * @throws ParseException
     */
    private function loadFromFile(string $filePath): array
    {
        return Yaml::parseFile($filePath);
    }

    private function applyInternalMutations(array $configuration): array
    {
        // parent configuration
        $parentConfiguration = $this->getParentConfiguration((array) ($configuration['Parent'] ?? []));

        // override parent configuration field
        foreach (self::FIELDS as $field) {
            if (isset($configuration[$field])) {
                $parentConfiguration[$field] = $configuration[$field];
            }
        }

        return $this->applyMutations($parentConfiguration, $configuration['Mutations'] ?? []);
    }

    private function applyAdditionalMutations(array $data): array
    {
        foreach ($this->additionalMutationFiles as $additionalMutationFile) {
            $mutation = $this->loadFromFile($additionalMutationFile);

            $data = $this->applyMutations($data, $mutation['Mutations'] ?? []);
        }

        return $data;
    }

    private function applyMutations(array $data, array $mutations): array
    {
        // apply common mutations
        $data = $this->applyMutation($data, $mutations);

        // apply optional mutations
        foreach (self::OPTIONS as $option) {
            if (isset($this->options[$option], $mutations[$option . '.' . $this->options[$option]])) {
                $data = $this->applyMutation($data, $mutations[$option . '.' . $this->options[$option]]);
            }
        }

        return $data;
    }

    private function applyMutation(array $data, array $mutation): array
    {
        foreach (self::FIELDS as $field) {
            $data[$field] = $this->applyMutationField($data[$field] ?? [], $field, $mutation);
        }

        return $data;
    }

    private function applyMutationField(array $data, string $field, array $mutation): array
    {
        if (isset($mutation['include'][$field])) {
            $data = array_merge($data, $mutation['include'][$field]);
        }

        if (isset($mutation['exclude'][$field])) {
            $data = array_diff($data, $mutation['exclude'][$field]);
        }

        return $data;
    }

    private function mergeConfiguration(array $configurations): array
    {
        $data = [];
        foreach (self::FIELDS as $field) {
            $data[$field] = $this->mergeConfigurationField($configurations, $field);
        }

        return $data;
    }

    private function mergeConfigurationField(array $configurations, string $field): array
    {
        $data = [];
        foreach ($configurations as $configuration) {
            $data[] = $configuration[$field] ?? [];
        }

        return $data ? array_merge(...$data) : [];
    }

    private function getParentConfiguration(array $filePaths): array
    {
        $parentConfigurations = [];
        foreach ($filePaths as $filePath) {
            $parentConfigurations[] = $this->getParentConfigurationFile($filePath);
        }

        return $this->mergeConfiguration($parentConfigurations);
    }

    private function getParentConfigurationFile(string $filePath): array
    {
        return (new self(dirname($this->filePath) . '/' . $filePath, $this->options))->getConfiguration();
    }
}
